/* SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB */
/* Copyright (c) 2021 Intel Corporation */
/*$FreeBSD$*/
#include <sys/param.h>
#include <sys/module.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/sysctl.h>
#include <sys/bus.h>
#include <machine/bus.h>
#include <linux/device.h>
#include <sys/rman.h>

#include "ice_rdma.h"
#include "main.h"

#include "irdma_if.h"
#include "irdma_di_if.h"

/**
 *  Driver version
 */
char		irdma_driver_version[] = "0.0.18";

/*
 * TUNABLE PARAMETERS:
 */
static SYSCTL_NODE(_hw, OID_AUTO, irdma, CTLFLAG_RD, 0, "irdma driver parameters");

int		debug = 0x7fffffff;
TUNABLE_INT("hw.irdma.debug", &debug);
SYSCTL_INT(_hw_irdma, OID_AUTO, debug, CTLFLAG_RWTUN,
	   &debug, 0, "irdma debug");

static void
irdma_init_tunable(struct irdma_pci_f *rf, uint8_t pf_id)
{
	struct sysctl_oid_list *irdma_sysctl_oid_list;
	char		pf_name[16];

	snprintf(pf_name, 15, "irdma%d", pf_id);
	sysctl_ctx_init(&rf->tun_info.irdma_sysctl_ctx);

	rf->tun_info.irdma_sysctl_tree = SYSCTL_ADD_NODE(&rf->tun_info.irdma_sysctl_ctx,
							 SYSCTL_STATIC_CHILDREN(_dev),
							 OID_AUTO, pf_name, CTLFLAG_RD,
							 NULL, "");

	irdma_sysctl_oid_list = SYSCTL_CHILDREN(rf->tun_info.irdma_sysctl_tree);

	/*
	 * roce/iwarp setting
	 */
	SYSCTL_ADD_U8(&rf->tun_info.irdma_sysctl_ctx, irdma_sysctl_oid_list, OID_AUTO,
		      "roce_enable", CTLFLAG_RDTUN, &rf->tun_info.roce_ena, 0,
		      "RoCEv2 mode enable");

	rf->protocol_used = IRDMA_IWARP_PROTOCOL_ONLY;
	if (rf->tun_info.roce_ena == 1)
		rf->protocol_used = IRDMA_ROCE_PROTOCOL_ONLY;
	else if (rf->tun_info.roce_ena != 0)
		printf("%s:%d wrong roce_enable value (%d), using iWARP\n",
		       __func__, __LINE__, rf->tun_info.roce_ena);
	printf("%s:%d protocol: %d %d %d\n", __func__, __LINE__,
	       rf->protocol_used, IRDMA_IWARP_PROTOCOL_ONLY,
	       IRDMA_ROCE_PROTOCOL_ONLY);
}

static struct irdma_handler *
irdma_find_handler(struct ice_rdma_peer *p_dev)
{
	struct irdma_handler *hdl;
	unsigned long	flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_for_each_entry(hdl, &irdma_handlers, list) {
		if (!hdl)
			continue;
		if (!hdl->rf.pdev)
			continue;
		if (hdl->rf.pdev->dev == p_dev->dev) {
			spin_unlock_irqrestore(&irdma_handler_lock, flags);
			return hdl;
		}
	}
	spin_unlock_irqrestore(&irdma_handler_lock, flags);

	return NULL;
}

/**
 * irdma_link_change - Callback for link state change
 * @peer: the peer interface structure
 * @linkstate: state of the link
 * @baudrate: speed of the link
 */
static void
irdma_link_change(struct ice_rdma_peer *peer, int linkstate, uint64_t baudrate)
{
	printf("%s:%d PF: %x, state: %d, speed: %lu\n", __func__, __LINE__,
	       peer->pf_id, linkstate, baudrate);
}

static void
irdma_fill_l2params(struct ice_rdma_peer *peer, struct irdma_l2params *l2params)
{
	int		i;

	l2params->mtu = peer->mtu;

	l2params->num_tc = peer->initial_qos_info.num_tc;
	l2params->num_apps = peer->initial_qos_info.num_apps;
	l2params->vsi_prio_type = peer->initial_qos_info.vsi_priority_type;
	l2params->vsi_rel_bw = peer->initial_qos_info.vsi_relative_bw;
	printf("%s:%d l2params: %d %d %d %d %d\n", __func__, __LINE__,
	       l2params->mtu, l2params->num_tc, l2params->num_apps,
	       l2params->vsi_prio_type, l2params->vsi_rel_bw);

	for (i = 0; i < l2params->num_tc; i++) {
		l2params->tc_info[i].egress_virt_up = peer->initial_qos_info.tc_info[i].egress_virt_up;
		l2params->tc_info[i].ingress_virt_up = peer->initial_qos_info.tc_info[i].ingress_virt_up;
		l2params->tc_info[i].prio_type = peer->initial_qos_info.tc_info[i].prio_type;
		l2params->tc_info[i].rel_bw = peer->initial_qos_info.tc_info[i].rel_bw;
		l2params->tc_info[i].tc_ctx = peer->initial_qos_info.tc_info[i].tc_ctx;
	}
	for (i = 0; i < ICE_TC_MAX_USER_PRIORITY; i++)
		l2params->up2tc[i] = peer->initial_qos_info.up2tc[i];
}

#if __FreeBSD_version < 1200000
/**
 * irdma_finalize_task - Finish open or close phase in a separate thread
 * @context: instance holding peer and iwdev information
 *
 * Triggered from irdma_open or irdma_close to perform rt_init_hw or
 * rt_deinit_hw respectively. Does registration and unregistration of
 * the device.
 */
static void
irdma_finalize_task(void *context, int pending)
{
	struct irdma_task_arg *task_arg = (struct irdma_task_arg *)context;
	struct irdma_device *iwdev = task_arg->iwdev;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_l2params l2params = {{{0}}};
	struct ice_rdma_peer *peer = task_arg->peer;
	int		status = 0;

	if (iwdev->iw_status) {
		irdma_ib_unregister_device(iwdev);
		irdma_rt_deinit_hw(iwdev);
		ib_dealloc_device(&iwdev->ibdev);

		IRDMA_DI_VSI_FILTER_UPDATE(peer, FALSE);
		printf("INIT: Gen2 VSI[%d] close complete ldev=%p\n",
		       peer->pf_vsi_num, peer);
	} else {
		irdma_fill_l2params(peer, &l2params);

		status = irdma_rt_init_hw(rf, iwdev, &l2params);
		if (status) {
			irdma_pr_err("RT init failed %d\n", status);
			ib_dealloc_device(&iwdev->ibdev);
			return;
		}

		status = irdma_ib_register_device(iwdev);
		if (status) {
			irdma_pr_err("Registration failed %d\n", status);
			irdma_rt_deinit_hw(iwdev);
			ib_dealloc_device(&iwdev->ibdev);
		}
	}
}

#endif
/**
 * irdma_open - Callback for operation open for RDMA device
 * @peer: the new peer interface structure
 *
 * Callback implementing the RDMA_OPEN function. Called by the ice driver to
 * notify the RDMA client driver that a new device has been initialized.
 */
static int
irdma_open(struct ice_rdma_peer *peer)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_sc_dev *dev;
	struct irdma_pci_f *rf;
	struct irdma_priv_ldev *pldev;
#if __FreeBSD_version >= 1200000
	struct irdma_l2params l2params = {{{0}}};
	enum irdma_status_code status;
#endif

	irdma_debug((struct irdma_sc_dev *)NULL, IRDMA_DEBUG_INIT, "begin %s", __FUNCTION__);
	hdl = irdma_find_handler(peer);
	if (!hdl)
		return -ENODEV;

	rf = &hdl->rf;
	if (rf->init_state != CEQ0_CREATED)
		return -EINVAL;

	iwdev = (struct irdma_device *)ib_alloc_device(sizeof(*iwdev));
	if (!iwdev)
		return -ENOMEM;

	iwdev->hdl = hdl;
	iwdev->rf = rf;
	iwdev->ldev = &rf->ldev;
	iwdev->netdev = rf->netdev;
	pldev = &rf->ldev;

	/* Set configfs default values */
	iwdev->push_mode = 0;
	iwdev->rcv_wnd = IRDMA_CM_DEFAULT_RCV_WND_SCALED;
	iwdev->rcv_wscale = IRDMA_CM_DEFAULT_RCV_WND_SCALE;
	iwdev->roce_cwnd = IRDMA_ROCE_CWND_DEFAULT;
	iwdev->roce_ackcreds = IRDMA_ROCE_ACKCREDS_DEFAULT;

	dev = &hdl->rf.sc_dev;
	if (rf->protocol_used == IRDMA_ROCE_PROTOCOL_ONLY) {
		iwdev->roce_mode = TRUE;
	}
	irdma_dev_info(rf->sc_dev, "irdma%d: protocol %s", peer->pf_id,
		       rf->protocol_used == IRDMA_ROCE_PROTOCOL_ONLY ? "RoCEv2" : "iWARP");

	iwdev->vsi_num = peer->pf_vsi_num;
	IRDMA_DI_VSI_FILTER_UPDATE(peer, TRUE);

#if __FreeBSD_version < 1200000
	rf->dev_ctx.task_arg.iwdev = iwdev;
	rf->dev_ctx.task_arg.peer = peer;

	taskqueue_enqueue(hdl->deferred_tq, &hdl->deferred_task);
#else
	irdma_fill_l2params(peer, &l2params);

	status = irdma_rt_init_hw(rf, iwdev, &l2params);
	if (status) {
		irdma_pr_err("RT init failed %d\n", status);
		ib_dealloc_device(&iwdev->ibdev);
		return -EIO;
	}

	status = irdma_ib_register_device(iwdev);
	if (status) {
		irdma_pr_err("Registration failed %d\n", status);
		irdma_rt_deinit_hw(iwdev);
		ib_dealloc_device(&iwdev->ibdev);
		return status;
	}
#endif

	return 0;
}

/**
 * irdma_close - Callback to notify that a peer device is down
 * @peer: the RDMA peer device being stopped
 *
 * Callback implementing the RDMA_CLOSE function. Called by the ice driver to
 * notify the RDMA client driver that a peer device is being stopped.
 */
static int
irdma_close(struct ice_rdma_peer *peer)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_pci_f *rf;

	hdl = irdma_find_handler(peer);
	if (!hdl)
		return IRDMA_ERR_LIST_EMPTY;

	rf = &hdl->rf;
	irdma_debug(&rf->sc_dev, IRDMA_DEBUG_INIT, "closing %s\n", __func__);
	iwdev = list_first_entry_or_null(&rf->vsi_dev_list, struct irdma_device,
					 list);
	if (!iwdev)
		return IRDMA_ERR_LIST_EMPTY;

#if __FreeBSD_version < 1200000
	taskqueue_enqueue(hdl->deferred_tq, &hdl->deferred_task);
#else
	irdma_ib_unregister_device(iwdev);
	irdma_rt_deinit_hw(iwdev);
	ib_dealloc_device(&iwdev->ibdev);

	IRDMA_DI_VSI_FILTER_UPDATE(peer, FALSE);
	printf("INIT: Gen2 VSI[%d] close complete ldev=%p\n",
	       peer->pf_vsi_num, peer);
#endif

	return 0;
}

/**
 * irdma_alloc_pcidev - allocate memory for pcidev and populate data
 * @peer: the new peer interface structure
 * @rf: RDMA PCI function
 */
static int
irdma_alloc_pcidev(struct ice_rdma_peer *peer, struct irdma_pci_f *rf)
{
	rf->pcidev = kzalloc(sizeof(struct pci_dev), GFP_KERNEL);
	if (!rf->pcidev) {
		return IRDMA_ERR_NO_MEMORY;
	}
#if __FreeBSD_version < 1202000
	rf->pcidev->vendor = pci_get_vendor(peer->dev);
	rf->pcidev->device = pci_get_device(peer->dev);
	rf->pcidev->subsystem_vendor = pci_get_subvendor(peer->dev);
	rf->pcidev->subsystem_device = pci_get_subdevice(peer->dev);
	rf->pcidev->revision = pci_get_revid(peer->dev);
#else
	if (linux_pci_attach_device(rf->dev_ctx.dev, NULL, NULL, rf->pcidev))
		return IRDMA_ERR_NO_MEMORY;
#endif

	return 0;
}

/**
 * irdma_dealloc_pcidev - deallocate memory for pcidev
 * @rf: RDMA PCI function
 */
static void
irdma_dealloc_pcidev(struct irdma_pci_f *rf)
{
#if __FreeBSD_version >= 1202000
	linux_pci_detach_device(rf->pcidev);
#endif
	kfree(rf->pcidev);
}

/**
 * irdma_probe - Callback to probe a new RDMA peer device
 * @peer: the new peer interface structure
 *
 * Callback implementing the RDMA_PROBE function. Called by the ice driver to
 * notify the RDMA client driver that a new device has been created
 */
int
irdma_probe(struct ice_rdma_peer *peer)
{
	device_printf(peer->dev, "%s: called\n", __func__);
	struct irdma_handler *hdl;
	struct irdma_pci_f *rf;
	struct irdma_sc_dev *dev;
	struct irdma_priv_ldev *pldev;
	struct ice_rdma_msix_mapping msix_info = {0};

	irdma_pr_info("probe: peer=%p, peer->pf_id=%d, peer->ifp=%p, peer->pci_mem->r_bustag=%lx\n",
		      peer, peer->pf_id, peer->ifp, peer->pci_mem->r_bustag);

	dump_struct(peer, sizeof(struct ice_rdma_peer), "ice_rdma_peer");

	hdl = irdma_find_handler(peer);

	if (hdl)
		return -EBUSY;

	hdl = kzalloc(sizeof(*hdl), GFP_KERNEL);
	if (!hdl)
		return IRDMA_ERR_NO_MEMORY;

	rf = &hdl->rf;
	pldev = &rf->ldev;
	hdl->ldev = pldev;
	rf->hdl = hdl;
	dev = &rf->sc_dev;
	dev->back_dev = rf;
	rf->gen_ops.init_hw = icrdma_init_hw;
	rf->gen_ops.register_qset = irdma_register_qset;
	rf->gen_ops.unregister_qset = irdma_unregister_qset;
	rf->pdev = peer;

	irdma_init_tunable(rf, peer->pf_id);

	rf->rdma_ver = IRDMA_GEN_2;
	rf->sc_dev.hw_attrs.uk_attrs.hw_rev = IRDMA_GEN_2;
	irdma_set_config_params(rf);
	rf->default_vsi.vsi_idx = peer->pf_vsi_num;
	pldev->pf_vsi_num = peer->pf_vsi_num;
	/* save information from ldev to priv_ldev */
	pldev->fn_num = peer->pf_id;
	rf->dev_ctx.dev = peer->dev;
	rf->dev_ctx.mem_bus_space_tag = rman_get_bustag(peer->pci_mem);
	rf->dev_ctx.mem_bus_space_handle = rman_get_bushandle(peer->pci_mem);
	rf->dev_ctx.mem_bus_space_size = rman_get_size(peer->pci_mem);
	irdma_debug(&rf->sc_dev, IRDMA_DEBUG_INIT, "VSI: %d", peer->pf_vsi_num);

	rf->hw.dev_context = &rf->dev_ctx;
	rf->hw.hw_addr = (u8 *)rman_get_virtual(peer->pci_mem);
	if (irdma_alloc_pcidev(peer, rf)) {
		kfree(hdl);
		return -ENOMEM;
	}
	rf->netdev = peer->ifp;
	pldev->ftype = FALSE;
	pldev->msix_count = peer->msix.count;
	pldev->msix_info.entry = peer->msix.base;
	pldev->msix_info.vector = peer->msix.count;
	printf("%s:%d msix_info: %d %d %d\n", __func__, __LINE__, pldev->msix_count, pldev->msix_info.entry, pldev->msix_info.vector);

	irdma_add_handler(hdl);

	/* allocate msixs */
	IRDMA_DI_MSIX_INIT(peer, &msix_info);

	irdma_find_handler(peer);
	if (irdma_ctrl_init_hw(rf)) {
		irdma_del_handler(hdl);
		irdma_dealloc_pcidev(rf);
		kfree(hdl);
		return -EIO;
	}

#if __FreeBSD_version < 1200000
	rf->dev_ctx.task_arg.peer = peer;

	TASK_INIT(&hdl->deferred_task, 0, irdma_finalize_task, &rf->dev_ctx.task_arg);
	hdl->deferred_tq = taskqueue_create_fast("irdma_defer",
						 M_NOWAIT, taskqueue_thread_enqueue, &hdl->deferred_tq);
	taskqueue_start_threads(&hdl->deferred_tq, 1, PI_NET, "irdma_defer_t");
#endif
	irdma_debug(&rf->sc_dev, IRDMA_DEBUG_INIT, "%s completed", __func__);
	/*
	 * XXX a proper RDMA driver would likely need to keep track of the peer structure in a list for future use.
	 */
	return 0;
}

/**
 * irdma_remove - Callback to remove an RDMA peer device
 * @peer: the new peer interface structure
 *
 * Callback implementing the RDMA_PROBE function. Called by the ice driver to
 * notify the RDMA client driver that a new device has been created
 */
int
irdma_remove(struct ice_rdma_peer *peer)
{
	struct irdma_handler *hdl;

	irdma_debug((struct irdma_sc_dev *)NULL, IRDMA_DEBUG_INIT, "removing %s\n", __FUNCTION__);

	hdl = irdma_find_handler(peer);
	if (!hdl)
		return 0;

#if __FreeBSD_version < 1200000
	taskqueue_drain(hdl->deferred_tq, &hdl->deferred_task);
	taskqueue_free(hdl->deferred_tq);
	hdl->rf.dev_ctx.task_arg.iwdev = NULL;
	hdl->rf.dev_ctx.task_arg.peer = NULL;
#endif
	printf("%s:%d removing\n", __func__, __LINE__);
	sysctl_ctx_free(&hdl->rf.tun_info.irdma_sysctl_ctx);
	hdl->rf.tun_info.irdma_sysctl_tree = NULL;
	irdma_ctrl_deinit_hw(&hdl->rf);
	irdma_del_handler(hdl);
	irdma_dealloc_pcidev(&hdl->rf);
	kfree(hdl);

	irdma_pr_info("IRDMA hardware deinitialization complete\n");

	return 0;
}

static void
irdma_prep_for_unregister()
{
	struct irdma_handler *hdl;
	unsigned long	flags;
	bool		hdl_valid;

	do {
		hdl_valid = FALSE;
		spin_lock_irqsave(&irdma_handler_lock, flags);
		list_for_each_entry(hdl, &irdma_handlers, list) {
			if (!hdl)
				continue;
			if (!hdl->rf.pdev)
				continue;
			hdl_valid = TRUE;
			break;
		}
		spin_unlock_irqrestore(&irdma_handler_lock, flags);
		if (!hdl || !hdl_valid)
			break;
		IRDMA_CLOSE(hdl->rf.pdev);
		IRDMA_REMOVE(hdl->rf.pdev);
	} while (1);
}

static kobj_method_t irdma_methods[] = {
	KOBJMETHOD(irdma_probe, irdma_probe),
	    KOBJMETHOD(irdma_open, irdma_open),
	    KOBJMETHOD(irdma_close, irdma_close),
	    KOBJMETHOD(irdma_remove, irdma_remove),
	    KOBJMETHOD(irdma_link_change, irdma_link_change),
	    KOBJMETHOD_END
};

/* declare irdma_class which extends the ice_rdma_di class */
DEFINE_CLASS_1(irdma, irdma_class, irdma_methods, sizeof(struct ice_rdma_peer), ice_rdma_di_class);

static struct ice_rdma_info irdma_info = {
	.major_version = ICE_RDMA_MAJOR_VERSION,
	.minor_version = ICE_RDMA_MINOR_VERSION,
	.patch_version = ICE_RDMA_PATCH_VERSION,
	.rdma_class = &irdma_class,
};

/**
 * irdma_module_event_handler - Module event handler callback
 * @mod: unused mod argument
 * @what: the module event to handle
 * @arg: unused module event argument
 *
 * Callback used by the FreeBSD module stack to notify the driver of module
 * events. Used to implement custom handling for certain module events such as
 * load and unload.
 */
static int
irdma_module_event_handler(module_t __unused mod, int what, void __unused * arg)
{
	switch (what) {
	case MOD_LOAD:
		printf("Loading irdma module\n");
		return ice_rdma_register(&irdma_info);
	case MOD_UNLOAD:
		printf("Unloading irdma module\n");
		irdma_prep_for_unregister();
		ice_rdma_unregister();
		return (0);
	default:
		return (EOPNOTSUPP);
	}

	return (0);
}

static moduledata_t irdma_moduledata = {
	"irdma",
	    irdma_module_event_handler,
	    NULL
};

DECLARE_MODULE(irdma, irdma_moduledata, SI_SUB_INIT_IF, SI_ORDER_ANY);
MODULE_VERSION(irdma, 1);
MODULE_DEPEND(irdma, ice, 1, 1, 1);
MODULE_DEPEND(irdma, ibcore, 1, 1, 1);
