// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2015 - 2021 Intel Corporation */
#include "main.h"

#define DRV_VER_MAJOR 1
#define DRV_VER_MINOR 4
#define DRV_VER_BUILD 24
#define DRV_VER	__stringify(DRV_VER_MAJOR) "."		\
	__stringify(DRV_VER_MINOR) "." __stringify(DRV_VER_BUILD)

static int resource_profile;
module_param(resource_profile, int, 0644);
MODULE_PARM_DESC(resource_profile, "Resource Profile: 0=PF only(default), 1=Weighted VF, 2=Even Distribution");

static int max_rdma_vfs = 32;
module_param(max_rdma_vfs, int, 0644);
MODULE_PARM_DESC(max_rdma_vfs, "Maximum VF count: 0-32, default=32");

bool irdma_upload_context;
module_param(irdma_upload_context, bool, 0644);
MODULE_PARM_DESC(irdma_upload_context, "Upload QP context, default=false");

static int limits_sel;
module_param(limits_sel, int, 0644);
MODULE_PARM_DESC(limits_sel, "Resource limits selector, Range: 0-7, default=0");

static int gen1_limits_sel = 2;
module_param(gen1_limits_sel, int, 0644);
MODULE_PARM_DESC(gen1_limits_sel, "x722 resource limits selector, Range: 0-5, default=2");

static int roce_ena;
module_param(roce_ena, int, 0644);
MODULE_PARM_DESC(roce_ena, "RoCE enable: 1=enable RoCEv2 on all ports (not supported on x722), 0=iWARP(default)");

static int en_rem_endpoint_trk;
module_param(en_rem_endpoint_trk, int, 0644);
MODULE_PARM_DESC(en_rem_endpoint_trk, "Remote Endpoint Tracking: 1=enabled (not supported on x722), 0=disabled(default)");

MODULE_ALIAS("i40iw");
MODULE_AUTHOR("Intel Corporation, <e1000-rdma@lists.sourceforge.net>");
MODULE_DESCRIPTION("Intel(R) Ethernet Protocol Driver for RDMA");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_VERSION(DRV_VER);

static struct notifier_block irdma_inetaddr_notifier = {
	.notifier_call = irdma_inetaddr_event
};

static struct notifier_block irdma_inetaddr6_notifier = {
	.notifier_call = irdma_inet6addr_event
};

static struct notifier_block irdma_net_notifier = {
	.notifier_call = irdma_net_event
};

static struct notifier_block irdma_netdevice_notifier = {
	.notifier_call = irdma_netdevice_event
};

static void irdma_register_notifiers(void)
{
	register_inetaddr_notifier(&irdma_inetaddr_notifier);
	register_inet6addr_notifier(&irdma_inetaddr6_notifier);
	register_netevent_notifier(&irdma_net_notifier);
	register_netdevice_notifier(&irdma_netdevice_notifier);
}

static void irdma_unregister_notifiers(void)
{
	unregister_netevent_notifier(&irdma_net_notifier);
	unregister_inetaddr_notifier(&irdma_inetaddr_notifier);
	unregister_inet6addr_notifier(&irdma_inetaddr6_notifier);
	unregister_netdevice_notifier(&irdma_netdevice_notifier);
}

/**
 * set_protocol_used - set protocol_used against HW generation and roce_ena flag
 * @rf: RDMA PCI function
 * @roce_ena: RoCE enabled flag
 */
static void set_protocol_used(struct irdma_pci_f *rf, bool roce_ena)
{
	switch (rf->rdma_ver) {
	case IRDMA_GEN_2:
		rf->protocol_used = roce_ena ? IRDMA_ROCE_PROTOCOL_ONLY :
					       IRDMA_IWARP_PROTOCOL_ONLY;
		break;
	case IRDMA_GEN_1:
		rf->protocol_used = IRDMA_IWARP_PROTOCOL_ONLY;
		break;
	}
}

void irdma_set_config_params(struct irdma_pci_f *rf)
{
	/* Setup RF configurations from module parameters */
#ifdef DEVLINK_SUPPORTED
	struct irdma_drvdata *drvdata = dev_get_drvdata(&rf->pdev->dev);
	struct irdma_dl_priv *dl_priv = devlink_priv(drvdata->dl);

	if (!dl_priv->reload) {
		rf->limits_sel = (rf->rdma_ver == IRDMA_GEN_1) ? gen1_limits_sel :
								 limits_sel;
		set_protocol_used(rf, roce_ena);
	} else {
		rf->limits_sel = dl_priv->limits_sel;
		set_protocol_used(rf, dl_priv->roce_ena);
	}
	rf->dl_priv = dl_priv;
#else
	rf->limits_sel = (rf->rdma_ver == IRDMA_GEN_1) ? gen1_limits_sel :
							 limits_sel;
	set_protocol_used(rf, roce_ena);
#endif /* DEVLINK_SUPPORTED */
	rf->rsrc_profile = (resource_profile < IRDMA_HMC_PROFILE_EQUAL) ?
			    (u8)resource_profile + IRDMA_HMC_PROFILE_DEFAULT :
			    IRDMA_HMC_PROFILE_DEFAULT;
	rf->max_rdma_vfs = (rf->rsrc_profile != IRDMA_HMC_PROFILE_DEFAULT) ?
			    max_rdma_vfs : 0;
	rf->max_ena_vfs = rf->max_rdma_vfs;
	rf->en_rem_endpoint_trk = en_rem_endpoint_trk;
	rf->rst_to = IRDMA_RST_TIMEOUT_HZ;
}

#ifdef DEVLINK_SUPPORTED
#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
static int irdma_devlink_rsrc_limits_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val)
#else
static int irdma_devlink_rsrc_limits_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val,
					      struct netlink_ext_ack *extack)
#endif
{
	u8 value = val.vu8;

	if (value > 7) {
		NL_SET_ERR_MSG_MOD(extack, "resource limits selector range is (0-7)");
		return -ERANGE;
	}

	return 0;
}

static int irdma_devlink_enable_roce_validate(struct devlink *dl, u32 id,
#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
					      union devlink_param_value val)
#else
					      union devlink_param_value val,
					      struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(dl);
	bool value = val.vbool;

	if (value && priv->drvdata->hw_ver == IRDMA_GEN_1) {
		NL_SET_ERR_MSG_MOD(extack, "RoCE not supported on device");
		return -EOPNOTSUPP;
	}

	return 0;
}

#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
static int irdma_devlink_fragcnt_limit_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val)
#else
static int irdma_devlink_fragcnt_limit_validate(struct devlink *dl, u32 id,
						union devlink_param_value val,
						struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(dl);
	u8 value = val.vu8;

	if (priv->drvdata->hw_ver == IRDMA_GEN_1) {
		NL_SET_ERR_MSG_MOD(extack, "feature not supproted on this device");
		return -EOPNOTSUPP;
	} else if (value > 13 || value < 2) {
		NL_SET_ERR_MSG_MOD(extack, "fragment count limit range (2-13)");
		return -ERANGE;
	}

	return 0;
}

static int irdma_devlink_upload_ctx_get(struct devlink *devlink, u32 id,
					struct devlink_param_gset_ctx *ctx)
{
	ctx->val.vbool = irdma_upload_context;
	return 0;
}

static int irdma_devlink_upload_ctx_set(struct devlink *devlink, u32 id,
					struct devlink_param_gset_ctx *ctx)
{
	irdma_upload_context = ctx->val.vbool;
	return 0;
}

static const struct devlink_param irdma_devlink_params[] = {
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
			     "resource_limits_selector", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_rsrc_limits_validate),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT,
			     "upload_context", DEVLINK_PARAM_TYPE_BOOL,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     irdma_devlink_upload_ctx_get,
			     irdma_devlink_upload_ctx_set, NULL),
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
			     "roce_enable", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_enable_roce_validate),
#else
	DEVLINK_PARAM_GENERIC(ENABLE_ROCE, BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_enable_roce_validate),
#endif
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
			     "dcqcn_enable", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
			     "dcqcn_cc_cfg_valid", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
			     "dcqcn_min_dec_factor", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_MIN_RATE,
			     "dcqcn_min_rate", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_F,
			     "dcqcn_F", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_T,
			     "dcqcn_T", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_B,
			     "dcqcn_B", DEVLINK_PARAM_TYPE_U32,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
			     "dcqcn_rai_factor", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
			     "dcqcn_hai_factor", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
			     "dcqcn_rreduce_mperiod", DEVLINK_PARAM_TYPE_U32,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
			     "fragment_count_limit", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_fragcnt_limit_validate),
};

static const struct devlink_ops irdma_devlink_ops = {
#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	.reload_up = irdma_devlink_reload_up,
	.reload_down = irdma_devlink_reload_down,
#ifdef IRDMA_DEVLINK_RELOAD_ACTIONS
	.reload_actions = BIT(DEVLINK_RELOAD_ACTION_DRIVER_REINIT),
#endif /* IRDMA_DEVLINK_RELOAD_ACTIONS */
#else
    	.reload = irdma_devlink_reload_up,
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
};

static void irdma_devlink_unregister(struct device *dev)
{
	struct irdma_drvdata *drvdata = dev_get_drvdata(dev);
	struct devlink *devlink = drvdata->dl;

#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	devlink_reload_disable(devlink);
#endif
	devlink_params_unregister(devlink, irdma_devlink_params,
				  ARRAY_SIZE(irdma_devlink_params));
	devlink_unregister(devlink);
	devlink_free(devlink);
}

static int irdma_devlink_register(struct device *dev, struct irdma_drvdata *drvdata)
{
	struct devlink *devlink;
	struct irdma_dl_priv *priv;
	union devlink_param_value value;
	int ret;

	devlink = devlink_alloc(&irdma_devlink_ops, sizeof(struct irdma_dl_priv));
	if (!devlink)
		return -ENOMEM;

	priv = devlink_priv(devlink);
	priv->limits_sel = (drvdata->hw_ver >= IRDMA_GEN_2) ? 0 : 2;
	priv->fragcnt_limit = (drvdata->hw_ver >= IRDMA_GEN_2) ? 6 : 2;
	priv->drvdata = drvdata;

	drvdata->dl = devlink;

	ret = devlink_register(devlink, dev);
	if (ret)
		goto err_dl_free;

	ret = devlink_params_register(devlink, irdma_devlink_params,
				      ARRAY_SIZE(irdma_devlink_params));
	if (ret)
		goto err_dl_unreg;

	value.vu8 = priv->limits_sel;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
				IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE, value);
#else
				DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE, value);
#endif
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_RATE, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_F, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_T, value);
	value.vu32 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_B, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR, value);
	value.vu32 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD, value);
	value.vu8 = priv->fragcnt_limit;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT, value);
	devlink_params_publish(devlink);
#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	devlink_reload_enable(devlink);
#endif

	return 0;

err_dl_unreg:
	devlink_unregister(devlink);
err_dl_free:
	devlink_free(devlink);

	return ret;
}

#endif /* DEVLINK_SUPPORTED */
static int irdma_probe(struct platform_device *pdev)
{
	struct irdma_drvdata *drvdata;
	const struct platform_device_id *id = platform_get_device_id(pdev);
	int ret;

	drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
	if (!drvdata)
		return -ENOMEM;

	switch (id->driver_data) {
	case IRDMA_GEN_2:
		drvdata->init_dev = irdma_init_dev;
		drvdata->deinit_dev = irdma_deinit_dev;
		break;
	case IRDMA_GEN_1:
		drvdata->init_dev = i40iw_init_dev;
		drvdata->deinit_dev = i40iw_deinit_dev;
		break;
	default:
		ret = -ENODEV;
		goto ver_err;
	}

	drvdata->hw_ver = id->driver_data;
#ifdef DEVLINK_SUPPORTED
	ret = irdma_devlink_register(&pdev->dev, drvdata);
	if (ret)
		goto ver_err;
#endif

	dev_set_drvdata(&pdev->dev, drvdata);
	ret = drvdata->init_dev(pdev);
	if (ret)
		goto probe_err;

	return 0;

probe_err:
#ifdef DEVLINK_SUPPORTED
	irdma_devlink_unregister(&pdev->dev);
#endif
ver_err:
	kfree(drvdata);

	return ret;
}

static int irdma_remove(struct platform_device *pdev)
{
	struct irdma_drvdata *drvdata = dev_get_drvdata(&pdev->dev);

	drvdata->deinit_dev(pdev);
#ifdef DEVLINK_SUPPORTED
	irdma_devlink_unregister(&pdev->dev);

#endif
	kfree(drvdata);
	return 0;
}

static int irdma_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct irdma_drvdata *drvdata = dev_get_drvdata(&pdev->dev);

	drvdata->deinit_dev(pdev);

	return 0;
}

static int irdma_resume(struct platform_device *pdev)
{
	struct irdma_drvdata *drvdata = dev_get_drvdata(&pdev->dev);

	return drvdata->init_dev(pdev);
}

static const struct platform_device_id irdma_platform_id_table[] = {
	{.name = IRDMA_ICE_PDEV_NAME, .driver_data = IRDMA_GEN_2},
	{.name = IRDMA_I40E_PDEV_NAME, .driver_data = IRDMA_GEN_1},
	{},
};

MODULE_DEVICE_TABLE(platform, irdma_platform_id_table);

static struct platform_driver irdma_pdriver = {
	.driver.name = "irdma",
	.id_table = irdma_platform_id_table,
	.probe = irdma_probe,
	.remove = irdma_remove,
	.resume = irdma_resume,
	.suspend = irdma_suspend,
};

/**
 * irdma_init_module - driver initialization function
 *
 * First function to call when the driver is loaded
 * Register the driver as ice client and port mapper client
 */
static int __init irdma_init_module(void)
{
	int ret;

	pr_info("irdma driver version: %d.%d.%d\n", DRV_VER_MAJOR,
		DRV_VER_MINOR, DRV_VER_BUILD);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_init();
#endif
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	ret = irdma_configfs_init();
	if (ret) {
		pr_err("Failed to register irdma to configfs subsystem\n");
#ifdef CONFIG_DEBUG_FS
		irdma_dbg_exit();
#endif
		return ret;
	}
#endif
	ret = platform_driver_register(&irdma_pdriver);
	if (ret) {
		pr_err("Failed irdma platform_driver_register()\n");
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
		irdma_configfs_exit();
#endif
#ifdef CONFIG_DEBUG_FS
		irdma_dbg_exit();
#endif
		return ret;
	}
	irdma_register_notifiers();

	return 0;
}

/**
 * irdma_exit_module - driver exit clean up function
 *
 * The function is called just before the driver is unloaded
 * Unregister the driver as ice client and port mapper client
 */
static void __exit irdma_exit_module(void)
{
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	irdma_configfs_exit();
#endif
	irdma_unregister_notifiers();
	platform_driver_unregister(&irdma_pdriver);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_exit();
#endif
}

module_init(irdma_init_module);
module_exit(irdma_exit_module);
