// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2018 - 2021 Intel Corporation */
#include "main.h"

struct dst_entry *irdma_get_fl6_dst(struct sockaddr_in6 *src_addr,
				    struct sockaddr_in6 *dst_addr)
{
	struct dst_entry *dst = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 39)
	struct flowi6 fl6 = {};

	fl6.daddr = dst_addr->sin6_addr;
	fl6.saddr = src_addr->sin6_addr;
	if (ipv6_addr_type(&fl6.daddr) & IPV6_ADDR_LINKLOCAL)
		fl6.flowi6_oif = dst_addr->sin6_scope_id;

	dst = ip6_route_output(&init_net, NULL, &fl6);
#else
	struct flowi fl = {};

	ipv6_addr_copy(&fl.fl6_dst, &dst_addr->sin6_addr);
	ipv6_addr_copy(&fl.fl6_src, &src_addr->sin6_addr);
	if (ipv6_addr_type(&fl.fl6_dst) & IPV6_ADDR_LINKLOCAL)
		fl.oif = dst_addr->sin6_scope_id;

	dst = ip6_route_output(&init_net, NULL, &fl);
#endif /* >= 2.6.39 */
	return dst;
}

struct neighbour *irdma_get_neigh_ipv6(struct dst_entry *dst,
				       struct sockaddr_in6 *dst_ipaddr)
{
	struct neighbour *neigh = NULL;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 1, 0)
	neigh = dst_neigh_lookup(dst, dst_ipaddr);
#elif LINUX_VERSION_CODE > KERNEL_VERSION(3, 0, 0)
	neigh = dst->_neighbour;
#else
	neigh = dst->neighbour;
#endif /* >= 3.1.0 */
	return neigh;
}

struct neighbour *irdma_get_neigh_ipv4(struct rtable *rt, __be32 *dst_ipaddr)
{
	struct neighbour *neigh = NULL;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 1, 0)
	neigh = dst_neigh_lookup(&rt->dst, dst_ipaddr);
#elif LINUX_VERSION_CODE > KERNEL_VERSION(3, 0, 0)
	neigh = rt->dst._neighbour;
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 39)
	neigh = rt->u.dst.neighbour;
#else
	neigh = rt->dst.neighbour;
#endif /* >= 3.1.0 */
	return neigh;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 9, 0)
struct net_device *irdma_netdev_master_upper_dev_get(struct net_device *netdev)
{
	return netdev->master;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 39)
struct rtable *irdma_ip_route_output(struct net *net, __be32 daddr,
				     __be32 saddr, u8 tos, int oif)
{
	struct flowi fl = {};
	struct rtable *rt;
	int ret;

	fl.nl_u.ip4_u.daddr = daddr;
	fl.nl_u.ip4_u.saddr = saddr;
	fl.oif = oif;
	ret = ip_route_output_key(net, &rt, &fl);
	return rt;
}

#endif /* 2.6.39 */
#endif /* 3.9.0 */
#ifdef IB_FW_VERSION_NAME_MAX
void irdma_get_dev_fw_str(struct ib_device *dev,
			  char *str)
{
	struct irdma_device *iwdev = to_iwdev(dev);

	snprintf(str, IB_FW_VERSION_NAME_MAX, "%u.%u",
		 FW_MAJOR_VER(&iwdev->rf->sc_dev),
		 FW_MINOR_VER(&iwdev->rf->sc_dev));
}
#else
void irdma_get_dev_fw_str(struct ib_device *dev,
			  char *str,
			  size_t str_len)
{
	struct irdma_device *iwdev = to_iwdev(dev);

	snprintf(str, str_len, "%u.%u",
		 FW_MAJOR_VER(&iwdev->rf->sc_dev),
		 FW_MINOR_VER(&iwdev->rf->sc_dev));
}
#endif /* IB_FW_VERSION_NAME_MAX */

#ifndef __OFED_BUILD__
#ifdef IRDMA_ADD_DEL_GID
int irdma_add_gid(struct ib_device *device,
		  u8 port_num,
		  unsigned int index,
		  const union ib_gid *gid,
		  const struct ib_gid_attr *attr,
		  void **context)
{
	return 0;
}

int irdma_del_gid(struct ib_device *device,
		  u8 port_num,
		  unsigned int index,
		  void **context)
{
	return 0;
}
#endif /* < 4.17.0 */
#endif /* OFED_BUILD */

#ifdef ALLOC_UCONTEXT_VER_2
/**
 * irdma_alloc_ucontext - Allocate the user context data structure
 * @uctx: context
 * @udata: user data
 *
 * This keeps track of all objects associated with a particular
 * user-mode client.
 */
int irdma_alloc_ucontext(struct ib_ucontext *uctx, struct ib_udata *udata)
{
	struct ib_device *ibdev = uctx->device;
	struct irdma_device *iwdev = to_iwdev(ibdev);
	struct irdma_alloc_ucontext_req req;
	struct irdma_alloc_ucontext_resp uresp = {};
	struct i40iw_alloc_ucontext_resp uresp_gen1 = {};
	struct irdma_ucontext *ucontext = to_ucontext(uctx);
	struct irdma_uk_attrs *uk_attrs;

	if (ib_copy_from_udata(&req, udata, min(sizeof(req), udata->inlen)))
		return -EINVAL;

	if (req.userspace_ver > IRDMA_ABI_VER)
		goto ver_error;

	ucontext->iwdev = iwdev;
	ucontext->abi_ver = req.userspace_ver;

	uk_attrs = &iwdev->rf->sc_dev.hw_attrs.uk_attrs;
	/* GEN_1 legacy support with libi40iw */
	if (req.userspace_ver <= 5) {
		if (uk_attrs->hw_rev != IRDMA_GEN_1)
			goto ver_error;

		uresp_gen1.max_qps = iwdev->rf->max_qp;
		uresp_gen1.max_pds = iwdev->rf->sc_dev.hw_attrs.max_hw_pds;
		uresp_gen1.wq_size = iwdev->rf->sc_dev.hw_attrs.max_qp_wr * 2;
		uresp_gen1.kernel_ver = req.userspace_ver;
		if (ib_copy_to_udata(udata, &uresp_gen1,
				     min(sizeof(uresp_gen1), udata->outlen)))
			return -EFAULT;
	} else {
		u64 bar_off =
		    (uintptr_t)iwdev->rf->sc_dev.hw_regs[IRDMA_DB_ADDR_OFFSET];
#ifdef RDMA_MMAP_DB_SUPPORT
		ucontext->db_mmap_entry =
			irdma_user_mmap_entry_insert(ucontext, bar_off,
						     IRDMA_MMAP_IO_NC,
						     &uresp.db_mmap_key);
#else
		spin_lock_init(&ucontext->mmap_tbl_lock);
		ucontext->db_mmap_entry =
			irdma_user_mmap_entry_add_hash(ucontext, bar_off,
						       IRDMA_MMAP_IO_NC,
						       &uresp.db_mmap_key);
#endif /* RDMA_MMAP_DB_SUPPORT */
		if (!ucontext->db_mmap_entry) {
#ifndef RDMA_MMAP_DB_SUPPORT
#endif
			return -ENOMEM;
		}
		uresp.kernel_ver = IRDMA_ABI_VER;
		uresp.feature_flags = uk_attrs->feature_flags;
		uresp.max_hw_wq_frags = uk_attrs->max_hw_wq_frags;
		uresp.max_hw_read_sges = uk_attrs->max_hw_read_sges;
		uresp.max_hw_inline = uk_attrs->max_hw_inline;
		uresp.max_hw_rq_quanta = uk_attrs->max_hw_rq_quanta;
		uresp.max_hw_wq_quanta = uk_attrs->max_hw_wq_quanta;
		uresp.max_hw_sq_chunk = uk_attrs->max_hw_sq_chunk;
		uresp.max_hw_cq_size = uk_attrs->max_hw_cq_size;
		uresp.min_hw_cq_size = uk_attrs->min_hw_cq_size;
		uresp.hw_rev = uk_attrs->hw_rev;
		if (ib_copy_to_udata(udata, &uresp,
				     min(sizeof(uresp), udata->outlen))) {
#ifdef RDMA_MMAP_DB_SUPPORT
			rdma_user_mmap_entry_remove(ucontext->db_mmap_entry);
#else
			irdma_user_mmap_entry_del_hash(ucontext->db_mmap_entry);
#endif
			return -EFAULT;
		}
	}

	INIT_LIST_HEAD(&ucontext->cq_reg_mem_list);
	spin_lock_init(&ucontext->cq_reg_mem_list_lock);
	INIT_LIST_HEAD(&ucontext->qp_reg_mem_list);
	spin_lock_init(&ucontext->qp_reg_mem_list_lock);
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 20, 0)
	INIT_LIST_HEAD(&ucontext->vma_list);
	mutex_init(&ucontext->vma_list_mutex);
#endif

#ifdef CONFIG_DEBUG_FS
	irdma_dbg_save_ucontext(iwdev, ucontext);
#endif
	return 0;

ver_error:
	dev_err(idev_to_dev(&iwdev->rf->sc_dev),
		"Invalid userspace driver version detected. Detected version %d, should be %d\n",
		req.userspace_ver, IRDMA_ABI_VER);
	return -EINVAL;
}
#endif

#ifdef ALLOC_UCONTEXT_VER_1
/**
 * irdma_alloc_ucontext - Allocate the user context data structure
 * @ibdev: ib device pointer
 * @udata: user data
 *
 * This keeps track of all objects associated with a particular
 * user-mode client.
 */
struct ib_ucontext *irdma_alloc_ucontext(struct ib_device *ibdev, struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(ibdev);
	struct irdma_alloc_ucontext_req req;
	struct irdma_alloc_ucontext_resp uresp = {};
	struct i40iw_alloc_ucontext_resp uresp_gen1 = {};
	struct irdma_ucontext *ucontext;
	struct irdma_uk_attrs *uk_attrs;

	if (ib_copy_from_udata(&req, udata, min(sizeof(req), udata->inlen)))
		return ERR_PTR(-EINVAL);

	if (req.userspace_ver > IRDMA_ABI_VER)
		goto ver_error;

	ucontext = kzalloc(sizeof(*ucontext), GFP_KERNEL);
	if (!ucontext)
		return ERR_PTR(-ENOMEM);

	ucontext->iwdev = iwdev;
	ucontext->abi_ver = req.userspace_ver;

	uk_attrs = &iwdev->rf->sc_dev.hw_attrs.uk_attrs;
	/* GEN_1 legacy support with libi40iw */
	if (req.userspace_ver <= 5) {
		if (uk_attrs->hw_rev != IRDMA_GEN_1) {
			kfree(ucontext);
			goto ver_error;
		}

		uresp_gen1.max_qps = iwdev->rf->max_qp;
		uresp_gen1.max_pds = iwdev->rf->sc_dev.hw_attrs.max_hw_pds;
		uresp_gen1.wq_size = iwdev->rf->sc_dev.hw_attrs.max_qp_wr * 2;
		uresp_gen1.kernel_ver = req.userspace_ver;
		if (ib_copy_to_udata(udata, &uresp_gen1,
				     min(sizeof(uresp_gen1), udata->outlen))) {
			kfree(ucontext);
			return ERR_PTR(-EFAULT);
		}
	} else {
		u64 bar_off =
		    (uintptr_t)iwdev->rf->sc_dev.hw_regs[IRDMA_DB_ADDR_OFFSET];

#ifdef RDMA_MMAP_DB_SUPPORT
		ucontext->db_mmap_entry =
			irdma_user_mmap_entry_insert(ucontext, bar_off,
						     IRDMA_MMAP_IO_NC,
						     &uresp.db_mmap_key);

#else
		spin_lock_init(&ucontext->mmap_tbl_lock);
		ucontext->db_mmap_entry =
			irdma_user_mmap_entry_add_hash(ucontext, bar_off,
						       IRDMA_MMAP_IO_NC,
						       &uresp.db_mmap_key);
#endif /* RDMA_MMAP_DB_SUPPORT */
		if (!ucontext->db_mmap_entry) {
#ifndef RDMA_MMAP_DB_SUPPORT
#endif
			kfree(ucontext);
			return ERR_PTR(-ENOMEM);
		}

		uresp.kernel_ver = IRDMA_ABI_VER;
		uresp.feature_flags = uk_attrs->feature_flags;
		uresp.max_hw_wq_frags = uk_attrs->max_hw_wq_frags;
		uresp.max_hw_read_sges = uk_attrs->max_hw_read_sges;
		uresp.max_hw_inline = uk_attrs->max_hw_inline;
		uresp.max_hw_rq_quanta = uk_attrs->max_hw_rq_quanta;
		uresp.max_hw_wq_quanta = uk_attrs->max_hw_wq_quanta;
		uresp.max_hw_sq_chunk = uk_attrs->max_hw_sq_chunk;
		uresp.max_hw_cq_size = uk_attrs->max_hw_cq_size;
		uresp.min_hw_cq_size = uk_attrs->min_hw_cq_size;
		uresp.hw_rev = uk_attrs->hw_rev;
		if (ib_copy_to_udata(udata, &uresp,
				     min(sizeof(uresp), udata->outlen))) {
#ifdef RDMA_MMAP_DB_SUPPORT
			rdma_user_mmap_entry_remove(ucontext->db_mmap_entry);
#else
			irdma_user_mmap_entry_del_hash(ucontext->db_mmap_entry);
#endif
			kfree(ucontext);
			return ERR_PTR(-EFAULT);
		}
	}

	INIT_LIST_HEAD(&ucontext->cq_reg_mem_list);
	spin_lock_init(&ucontext->cq_reg_mem_list_lock);
	INIT_LIST_HEAD(&ucontext->qp_reg_mem_list);
	spin_lock_init(&ucontext->qp_reg_mem_list_lock);
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 20, 0)
	INIT_LIST_HEAD(&ucontext->vma_list);
	mutex_init(&ucontext->vma_list_mutex);
#endif

#ifdef CONFIG_DEBUG_FS
	irdma_dbg_save_ucontext(iwdev, ucontext);
#endif
	return &ucontext->ibucontext;

ver_error:
	ibdev_err(&iwdev->ibdev,
		  "Invalid userspace driver version detected. Detected version %d, should be %d\n",
		  req.userspace_ver, IRDMA_ABI_VER);
	return ERR_PTR(-EINVAL);
}
#endif

#ifdef DEALLOC_UCONTEXT_VER_2
/**
 * irdma_dealloc_ucontext - deallocate the user context data structure
 * @context: user context created during alloc
 */
void irdma_dealloc_ucontext(struct ib_ucontext *context)
{
	struct irdma_ucontext *ucontext = to_ucontext(context);

#ifdef RDMA_MMAP_DB_SUPPORT
	rdma_user_mmap_entry_remove(ucontext->db_mmap_entry);
#else
	irdma_user_mmap_entry_del_hash(ucontext->db_mmap_entry);
#endif
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_free_ucontext(ucontext);
#endif

	return;
}
#endif

#ifdef DEALLOC_UCONTEXT_VER_1
/**
 * irdma_dealloc_ucontext - deallocate the user context data structure
 * @context: user context created during alloc
 */
int irdma_dealloc_ucontext(struct ib_ucontext *context)
{
	struct irdma_ucontext *ucontext = to_ucontext(context);

#ifdef RDMA_MMAP_DB_SUPPORT
	rdma_user_mmap_entry_remove(ucontext->db_mmap_entry);
#else
	irdma_user_mmap_entry_del_hash(ucontext->db_mmap_entry);
#endif
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_free_ucontext(ucontext);
#endif
	kfree(ucontext);

	return 0;
}
#endif

#ifdef ALLOC_PD_VER_3
/**
 * irdma_alloc_pd - allocate protection domain
 * @pd: protection domain
 * @udata: user data
 */
int irdma_alloc_pd(struct ib_pd *pd, struct ib_udata *udata)
{
	struct irdma_pd *iwpd = to_iwpd(pd);
	struct irdma_device *iwdev = to_iwdev(pd->device);
	struct irdma_sc_dev *dev = &iwdev->rf->sc_dev;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_alloc_pd_resp uresp = {};
	struct irdma_sc_pd *sc_pd;
	u32 pd_id = 0;
	int err;

	err = irdma_alloc_rsrc(rf, rf->allocated_pds, rf->max_pd, &pd_id,
			       &rf->next_pd);
	if (err)
		return err;

	sc_pd = &iwpd->sc_pd;
	if (udata) {
		struct irdma_ucontext *ucontext =
			rdma_udata_to_drv_context(udata, struct irdma_ucontext,
						  ibucontext);

		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, ucontext->abi_ver);
		uresp.pd_id = pd_id;
		if (ib_copy_to_udata(udata, &uresp,
				     min(sizeof(uresp), udata->outlen))) {
			err = -EFAULT;
			goto error;
		}
	} else {
		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, IRDMA_ABI_VER);
	}

		return 0;

error:

	irdma_free_rsrc(rf, rf->allocated_pds, pd_id);

	return err;
}
#endif

#ifdef ALLOC_PD_VER_2
/**
 * irdma_alloc_pd - allocate protection domain
 * @pd: protection domain
 * @context: user context
 * @udata: user data
 */
int irdma_alloc_pd(struct ib_pd *pd, struct ib_ucontext *context, struct ib_udata *udata)
{
	struct irdma_pd *iwpd = to_iwpd(pd);
	struct irdma_device *iwdev = to_iwdev(pd->device);
	struct irdma_sc_dev *dev = &iwdev->rf->sc_dev;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_alloc_pd_resp uresp = {};
	struct irdma_sc_pd *sc_pd;
	u32 pd_id = 0;
	int err;

	err = irdma_alloc_rsrc(rf, rf->allocated_pds, rf->max_pd, &pd_id,
			       &rf->next_pd);
	if (err)
		return err;

	sc_pd = &iwpd->sc_pd;
	if (udata) {
		struct irdma_ucontext *ucontext = to_ucontext(context);

		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, ucontext->abi_ver);
		uresp.pd_id = pd_id;
		if (ib_copy_to_udata(udata, &uresp,
				     min(sizeof(uresp), udata->outlen))) {
			err = -EFAULT;
			goto error;
		}
	} else {
		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, IRDMA_ABI_VER);
	}

		return 0;

error:

	irdma_free_rsrc(rf, rf->allocated_pds, pd_id);

	return err;
}
#endif

#ifdef ALLOC_PD_VER_1
/**
 * irdma_alloc_pd - allocate protection domain
 * @ibdev: IB device
 * @context: user context
 * @udata: user data
 */
struct ib_pd *irdma_alloc_pd(struct ib_device *ibdev, struct ib_ucontext *context, struct ib_udata *udata)
{
	struct irdma_pd *iwpd;
	struct irdma_device *iwdev = to_iwdev(ibdev);
	struct irdma_sc_dev *dev = &iwdev->rf->sc_dev;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_alloc_pd_resp uresp = {};
	struct irdma_sc_pd *sc_pd;
	u32 pd_id = 0;
	int err;

	err = irdma_alloc_rsrc(rf, rf->allocated_pds, rf->max_pd, &pd_id,
			       &rf->next_pd);
	if (err)
		return ERR_PTR(err);

	iwpd = kzalloc(sizeof(*iwpd), GFP_KERNEL);
	if (!iwpd) {
		err = -ENOMEM;
		goto free_res;
	}

	sc_pd = &iwpd->sc_pd;
	if (udata) {
		struct irdma_ucontext *ucontext = to_ucontext(context);

		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, ucontext->abi_ver);
		uresp.pd_id = pd_id;
		if (ib_copy_to_udata(udata, &uresp,
				     min(sizeof(uresp), udata->outlen))) {
			err = -EFAULT;
			goto error;
		}
	} else {
		dev->iw_pd_ops->pd_init(dev, sc_pd, pd_id, IRDMA_ABI_VER);
	}

		return &iwpd->ibpd;

error:
	kfree(iwpd);
free_res:

	irdma_free_rsrc(rf, rf->allocated_pds, pd_id);

	return ERR_PTR(err);
}

#endif

#ifdef DEALLOC_PD_VER_4
int irdma_dealloc_pd(struct ib_pd *ibpd, struct ib_udata *udata)
{
	struct irdma_pd *iwpd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);

	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_pds, iwpd->sc_pd.pd_id);
	return 0;
}

#endif

#ifdef DEALLOC_PD_VER_3
void irdma_dealloc_pd(struct ib_pd *ibpd, struct ib_udata *udata)
{
	struct irdma_pd *iwpd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);

	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_pds, iwpd->sc_pd.pd_id);
}

#endif

#ifdef DEALLOC_PD_VER_2
void irdma_dealloc_pd(struct ib_pd *ibpd)
{
	struct irdma_pd *iwpd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);

	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_pds, iwpd->sc_pd.pd_id);
}
#endif

#ifdef DEALLOC_PD_VER_1
int irdma_dealloc_pd(struct ib_pd *ibpd)
{
	struct irdma_pd *iwpd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);

	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_pds, iwpd->sc_pd.pd_id);
	kfree(iwpd);
	return 0;
}
#endif

#ifdef CREATE_AH_VER_3
/**
 * irdma_create_ah - create address handle
 * @ibpd: Protection Domain for AH
 * @attr: address handle attributes
 * @flags: AH flags to wait
 * @udata: user data
 *
 * returns a pointer to an address handle
 */
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      u32 flags,
			      struct ib_udata *udata)
{
	struct irdma_pd *pd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);
	struct irdma_ah *ah;
	const struct ib_gid_attr *sgid_attr;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_sc_ah *sc_ah;
	u32 ah_id = 0;
	struct irdma_ah_info *ah_info;
	struct irdma_create_ah_resp uresp;
	union {
		struct sockaddr saddr;
		struct sockaddr_in saddr_in;
		struct sockaddr_in6 saddr_in6;
	} sgid_addr, dgid_addr;
	int err;
	u8 dmac[ETH_ALEN];
	bool sleep;

	err = irdma_alloc_rsrc(rf, rf->allocated_ahs,
			       rf->max_ah, &ah_id, &rf->next_ah);
	if (err)
		return ERR_PTR(err);

	ah = kzalloc(sizeof(*ah), GFP_ATOMIC);
	if (!ah) {
		irdma_free_rsrc(rf, rf->allocated_ahs, ah_id);
		return ERR_PTR(-ENOMEM);
	}

	ah->pd = pd;
	sc_ah = &ah->sc_ah;
	sc_ah->ah_info.ah_idx = ah_id;
	sc_ah->ah_info.vsi = &iwdev->vsi;
	iwdev->rf->sc_dev.iw_uda_ops->init_ah(&rf->sc_dev, sc_ah);
	ah->sgid_index = attr->grh.sgid_index;
	memcpy(&ah->dgid, &attr->grh.dgid, sizeof(ah->dgid));
	sgid_attr = attr->grh.sgid_attr;
	rdma_gid2ip((struct sockaddr *)&sgid_addr, &sgid_attr->gid);
	rdma_gid2ip((struct sockaddr *)&dgid_addr, &attr->grh.dgid);
	ah->av.attrs = *attr;
	ah->av.net_type = kc_rdma_gid_attr_network_type(sgid_attr,
							sgid_attr.gid_type,
							&sgid);
	ah->av.sgid_addr.saddr = sgid_addr.saddr;
	ah->av.dgid_addr.saddr = dgid_addr.saddr;
	ah_info = &sc_ah->ah_info;
	ah_info->ah_idx = ah_id;
	ah_info->pd_idx = pd->sc_pd.pd_id;
	ether_addr_copy(ah_info->mac_addr, iwdev->netdev->dev_addr);
	if (attr->ah_flags & IB_AH_GRH) {
		ah_info->flow_label = attr->grh.flow_label;
		ah_info->hop_ttl = attr->grh.hop_limit;
		ah_info->tc_tos = attr->grh.traffic_class;
	}
	ether_addr_copy(dmac, attr->roce.dmac);

	if (kc_rdma_gid_attr_network_type(sgid_attr, sgid_attr.gid_type,
					  &sgid) == RDMA_NETWORK_IPV4) {
		ah_info->ipv4_valid = true;
		ah_info->dest_ip_addr[0] =
			ntohl(dgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->src_ip_addr[0] =
			ntohl(sgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->do_lpbk = irdma_ipv4_is_lpb(ah_info->src_ip_addr[0],
					ah_info->dest_ip_addr[0]);
		if (ipv4_is_multicast(dgid_addr.saddr_in.sin_addr.s_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, true);
	} else {
		irdma_copy_ip_ntohl(ah_info->dest_ip_addr,
				dgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		irdma_copy_ip_ntohl(ah_info->src_ip_addr,
				sgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		ah_info->do_lpbk = irdma_ipv6_is_lpb(ah_info->src_ip_addr,
						     ah_info->dest_ip_addr);
		if (rdma_is_multicast_addr(&dgid_addr.saddr_in6.sin6_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, false);
	}

	if (kc_deref_sgid_attr(sgid_attr) && is_vlan_dev(kc_deref_sgid_attr(sgid_attr)))
		ah_info->vlan_tag = vlan_dev_vlan_id(kc_deref_sgid_attr(sgid_attr));
	else
		ah_info->vlan_tag = VLAN_N_VID;

	ah_info->dst_arpindex = irdma_add_arp(iwdev->rf, ah_info->dest_ip_addr,
					      ah_info->ipv4_valid, dmac);

	if (ah_info->dst_arpindex == -1) {
		err = -EINVAL;
		goto error;
	}

	if (ah_info->vlan_tag >= VLAN_N_VID && iwdev->dcb)
		ah_info->vlan_tag = 0;

	if (ah_info->vlan_tag < VLAN_N_VID) {
		ah_info->insert_vlan_tag = true;
		ah_info->vlan_tag |=
		rt_tos2priority(ah_info->tc_tos) << VLAN_PRIO_SHIFT;
	}

	if (udata) {
		mutex_lock(&iwdev->ah_list_lock);
		if (irdma_ah_exists(iwdev, ah)) {
			mutex_unlock(&iwdev->ah_list_lock);
			irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs,
					ah_id);
#ifdef CONFIG_DEBUG_FS
			iwdev->ah_reused++;
#endif
			goto copy_udata;
		}
	}

	sleep = flags & RDMA_CREATE_AH_SLEEPABLE;

	err = irdma_ah_cqp_op(iwdev->rf, sc_ah, IRDMA_OP_AH_CREATE,
			      sleep, irdma_gsi_ud_qp_ah_cb, sc_ah);
	if (err) {
		if (udata) {
			if (ah->parent_ah) {
				list_del(&ah->parent_ah->list);
				kfree(ah->parent_ah);
				iwdev->ah_list_cnt--;
			}
			mutex_unlock(&iwdev->ah_list_lock);
		}
		irdma_dbg(iwdev_to_idev(iwdev),
			  "VERBS: CQP-OP Create AH fail");
		goto error;
	}

	if (udata)
		mutex_unlock(&iwdev->ah_list_lock);

	if (!sleep) {
		int cnt = CQP_COMPL_WAIT_TIME_MS * CQP_TIMEOUT_THRESHOLD;

		do {
			irdma_cqp_ce_handler(rf, &rf->ccq.sc_cq);
			mdelay(1);
		} while (!sc_ah->ah_info.ah_valid && --cnt);

		if (!cnt) {
			irdma_dbg(iwdev_to_idev(iwdev),
				  "VERBS: CQP create AH timed out");
			err = -ETIMEDOUT;
			goto error;
		}
	}

copy_udata:
	if (udata) {
		uresp.ah_id = ah->sc_ah.ah_info.ah_idx;
		err = ib_copy_to_udata(udata, &uresp, sizeof(uresp));
	}

	return &ah->ibah;

error:
		kfree(ah);
		irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);

		return ERR_PTR(err);
	}
#endif

#ifdef CREATE_AH_VER_4
/**
 * irdma_create_ah - create address handle
 * @ibpd: ptr to pd
 * @attr: address handle attributes
 * @udata: user data
 *
 * returns a pointer to an address handle
 */
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      struct ib_udata *udata)
{
	struct irdma_pd *pd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);
	struct irdma_ah *ah;
	const struct ib_gid_attr *sgid_attr;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_sc_ah *sc_ah;
	u32 ah_id = 0;
	struct irdma_ah_info *ah_info;
	struct irdma_create_ah_resp uresp;
	union {
		struct sockaddr saddr;
		struct sockaddr_in saddr_in;
		struct sockaddr_in6 saddr_in6;
	} sgid_addr, dgid_addr;
	int err;
	u8 dmac[ETH_ALEN];
	bool sleep;

	err = irdma_alloc_rsrc(rf, rf->allocated_ahs,
			       rf->max_ah, &ah_id, &rf->next_ah);
	if (err)
		return ERR_PTR(err);

	ah = kzalloc(sizeof(*ah), GFP_ATOMIC);
	if (!ah) {
		irdma_free_rsrc(rf, rf->allocated_ahs, ah_id);
		return ERR_PTR(-ENOMEM);
	}

	ah->pd = pd;
	sc_ah = &ah->sc_ah;
	sc_ah->ah_info.ah_idx = ah_id;
	sc_ah->ah_info.vsi = &iwdev->vsi;
	iwdev->rf->sc_dev.iw_uda_ops->init_ah(&rf->sc_dev, sc_ah);
	ah->sgid_index = attr->grh.sgid_index;
	memcpy(&ah->dgid, &attr->grh.dgid, sizeof(ah->dgid));
	sgid_attr = attr->grh.sgid_attr;
	rdma_gid2ip((struct sockaddr *)&sgid_addr, &sgid_attr->gid);
	rdma_gid2ip((struct sockaddr *)&dgid_addr, &attr->grh.dgid);
	ah->av.attrs = *attr;
	ah->av.net_type = kc_rdma_gid_attr_network_type(sgid_attr,
							sgid_attr.gid_type,
							&sgid);
	ah->av.sgid_addr.saddr = sgid_addr.saddr;
	ah->av.dgid_addr.saddr = dgid_addr.saddr;
	ah_info = &sc_ah->ah_info;
	ah_info->ah_idx = ah_id;
	ah_info->pd_idx = pd->sc_pd.pd_id;
	ether_addr_copy(ah_info->mac_addr, iwdev->netdev->dev_addr);
	if (attr->ah_flags & IB_AH_GRH) {
		ah_info->flow_label = attr->grh.flow_label;
		ah_info->hop_ttl = attr->grh.hop_limit;
		ah_info->tc_tos = attr->grh.traffic_class;
	}
	ether_addr_copy(dmac, attr->roce.dmac);

	if (kc_rdma_gid_attr_network_type(sgid_attr, sgid_attr.gid_type,
					  &sgid) == RDMA_NETWORK_IPV4) {
		ah_info->ipv4_valid = true;
		ah_info->dest_ip_addr[0] =
			ntohl(dgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->src_ip_addr[0] =
			ntohl(sgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->do_lpbk = irdma_ipv4_is_lpb(ah_info->src_ip_addr[0],
					ah_info->dest_ip_addr[0]);
		if (ipv4_is_multicast(dgid_addr.saddr_in.sin_addr.s_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, true);
	} else {
		irdma_copy_ip_ntohl(ah_info->dest_ip_addr,
				dgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		irdma_copy_ip_ntohl(ah_info->src_ip_addr,
				sgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		ah_info->do_lpbk = irdma_ipv6_is_lpb(ah_info->src_ip_addr,
						     ah_info->dest_ip_addr);
		if (rdma_is_multicast_addr(&dgid_addr.saddr_in6.sin6_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, false);
	}

	if (kc_deref_sgid_attr(sgid_attr) && is_vlan_dev(kc_deref_sgid_attr(sgid_attr)))
		ah_info->vlan_tag = vlan_dev_vlan_id(kc_deref_sgid_attr(sgid_attr));
	else
		ah_info->vlan_tag = VLAN_N_VID;

	ah_info->dst_arpindex = irdma_add_arp(iwdev->rf, ah_info->dest_ip_addr,
					      ah_info->ipv4_valid, dmac);

	if (ah_info->dst_arpindex == -1) {
		err = -EINVAL;
		goto error;
	}

	if (ah_info->vlan_tag >= VLAN_N_VID && iwdev->dcb)
		ah_info->vlan_tag = 0;

	if (ah_info->vlan_tag < VLAN_N_VID) {
		ah_info->insert_vlan_tag = true;
		ah_info->vlan_tag |=
		rt_tos2priority(ah_info->tc_tos) << VLAN_PRIO_SHIFT;
	}

	if (udata) {
		mutex_lock(&iwdev->ah_list_lock);
		if (irdma_ah_exists(iwdev, ah)) {
			mutex_unlock(&iwdev->ah_list_lock);
			irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs,
					ah_id);
#ifdef CONFIG_DEBUG_FS
			iwdev->ah_reused++;
#endif
			goto copy_udata;
		}
	}

	sleep = (udata ? true : false);

	err = irdma_ah_cqp_op(iwdev->rf, sc_ah, IRDMA_OP_AH_CREATE,
			      sleep, irdma_gsi_ud_qp_ah_cb, sc_ah);
	if (err) {
		if (udata) {
			if  (ah->parent_ah) {
				list_del(&ah->parent_ah->list);
				kfree(ah->parent_ah);
				iwdev->ah_list_cnt--;
			}
			mutex_unlock(&iwdev->ah_list_lock);
		}
		irdma_dbg(iwdev_to_idev(iwdev),
			  "VERBS: CQP-OP Create AH fail");
		goto error;
	}

	if (udata)
		mutex_unlock(&iwdev->ah_list_lock);

	if (!sleep) {
		int cnt = CQP_COMPL_WAIT_TIME_MS * CQP_TIMEOUT_THRESHOLD;

		do {
			irdma_cqp_ce_handler(rf, &rf->ccq.sc_cq);
			mdelay(1);
		} while (!sc_ah->ah_info.ah_valid && --cnt);

		if (!cnt) {
			irdma_dbg(iwdev_to_idev(iwdev),
				  "VERBS: CQP create AH timed out");
			err = -ETIMEDOUT;
			goto error;
		}
	}

copy_udata:
	if (udata) {
		uresp.ah_id = ah->sc_ah.ah_info.ah_idx;
		err = ib_copy_to_udata(udata, &uresp, sizeof(uresp));
	}

	return &ah->ibah;

error:
		kfree(ah);
		irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);

		return ERR_PTR(err);
	}
#endif

#ifdef CREATE_AH_VER_2
/**
 * irdma_create_ah - create address handle
 * @ib_ah: ptr to AH
 * @attr: address handle attributes
 * @flags: AH flags to wait
 * @udata: user data
 *
 * returns 0 on success, error otherwise
 */
int irdma_create_ah(struct ib_ah *ib_ah,
		    struct rdma_ah_attr *attr, u32 flags,
		    struct ib_udata *udata)
#elif defined(CREATE_AH_VER_5)
int irdma_create_ah_v2(struct ib_ah *ib_ah,
		       struct rdma_ah_attr *attr, u32 flags,
		       struct ib_udata *udata)
#endif
#if defined(CREATE_AH_VER_2) || defined(CREATE_AH_VER_5)
{
	struct irdma_pd *pd = to_iwpd(ib_ah->pd);
	struct irdma_ah *ah = container_of(ib_ah, struct irdma_ah, ibah);
	struct irdma_device *iwdev = to_iwdev(ib_ah->pd->device);
	const struct ib_gid_attr *sgid_attr;
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_sc_ah *sc_ah;
	u32 ah_id = 0;
	struct irdma_ah_info *ah_info;
	struct irdma_create_ah_resp uresp;
	union {
		struct sockaddr saddr;
		struct sockaddr_in saddr_in;
		struct sockaddr_in6 saddr_in6;
	} sgid_addr, dgid_addr;
	int err;
	u8 dmac[ETH_ALEN];
	bool sleep;

	err = irdma_alloc_rsrc(rf, rf->allocated_ahs,
			       rf->max_ah, &ah_id, &rf->next_ah);

	if (err)
		return err;

	ah->pd = pd;
	sc_ah = &ah->sc_ah;
	sc_ah->ah_info.ah_idx = ah_id;
	sc_ah->ah_info.vsi = &iwdev->vsi;
	iwdev->rf->sc_dev.iw_uda_ops->init_ah(&rf->sc_dev, sc_ah);
	ah->sgid_index = attr->grh.sgid_index;
	memcpy(&ah->dgid, &attr->grh.dgid, sizeof(ah->dgid));
	sgid_attr = attr->grh.sgid_attr;

	rdma_gid2ip((struct sockaddr *)&sgid_addr, &sgid_attr->gid);
	rdma_gid2ip((struct sockaddr *)&dgid_addr, &attr->grh.dgid);
	ah->av.attrs = *attr;
	ah->av.net_type = kc_rdma_gid_attr_network_type(sgid_attr,
							sgid_attr.gid_type,
							&sgid);

	ah->av.sgid_addr.saddr = sgid_addr.saddr;
	ah->av.dgid_addr.saddr = dgid_addr.saddr;
	ah_info = &sc_ah->ah_info;
	ah_info->ah_idx = ah_id;
	ah_info->pd_idx = pd->sc_pd.pd_id;
	err = rdma_read_gid_l2_fields(sgid_attr, &ah_info->vlan_tag,
				      ah_info->mac_addr);

	if (err)
		goto error;

	if (attr->ah_flags & IB_AH_GRH) {
		ah_info->flow_label = attr->grh.flow_label;
		ah_info->hop_ttl = attr->grh.hop_limit;
		ah_info->tc_tos = attr->grh.traffic_class;
	}

	ether_addr_copy(dmac, attr->roce.dmac);

	if (kc_rdma_gid_attr_network_type(sgid_attr, sgid_attr.gid_type,
					  &sgid) == RDMA_NETWORK_IPV4) {
		ah_info->ipv4_valid = true;
		ah_info->dest_ip_addr[0] =
				ntohl(dgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->src_ip_addr[0] =
				ntohl(sgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->do_lpbk = irdma_ipv4_is_lpb(ah_info->src_ip_addr[0],
						     ah_info->dest_ip_addr[0]);
		if (ipv4_is_multicast(dgid_addr.saddr_in.sin_addr.s_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, true);
	} else {
		irdma_copy_ip_ntohl(ah_info->dest_ip_addr,
				    dgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		irdma_copy_ip_ntohl(ah_info->src_ip_addr,
				    sgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		ah_info->do_lpbk = irdma_ipv6_is_lpb(ah_info->src_ip_addr,
						     ah_info->dest_ip_addr);
		if (rdma_is_multicast_addr(&dgid_addr.saddr_in6.sin6_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, false);
	}

	ah_info->dst_arpindex = irdma_add_arp(iwdev->rf, ah_info->dest_ip_addr,
					      ah_info->ipv4_valid, dmac);

	if (ah_info->dst_arpindex == -1) {
		err = -EINVAL;
		goto error;
	}

	if (ah_info->vlan_tag >= VLAN_N_VID && iwdev->dcb)
		ah_info->vlan_tag = 0;

	if (ah_info->vlan_tag < VLAN_N_VID) {
		ah_info->insert_vlan_tag = true;
		ah_info->vlan_tag |=
			rt_tos2priority(ah_info->tc_tos) << VLAN_PRIO_SHIFT;
	}

	if (udata) {
		mutex_lock(&iwdev->ah_list_lock);
		if (irdma_ah_exists(iwdev, ah)) {
			mutex_unlock(&iwdev->ah_list_lock);
			irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);
#ifdef CONFIG_DEBUG_FS
			iwdev->ah_reused++;
#endif
			goto copy_udata;
		}
	}

	sleep = flags & RDMA_CREATE_AH_SLEEPABLE;

	err = irdma_ah_cqp_op(iwdev->rf, sc_ah, IRDMA_OP_AH_CREATE,
			      sleep, irdma_gsi_ud_qp_ah_cb, sc_ah);
	if (err) {
		if (udata) {
			if (ah->parent_ah) {
				list_del(&ah->parent_ah->list);
				kfree(ah->parent_ah);
				iwdev->ah_list_cnt--;
			}
			mutex_unlock(&iwdev->ah_list_lock);
		}
		irdma_dbg(iwdev_to_idev(iwdev),
			  "VERBS: CQP-OP Create AH fail");
		goto error;
	}

	if (udata)
		mutex_unlock(&iwdev->ah_list_lock);

	if (!sleep) {
		int cnt = CQP_COMPL_WAIT_TIME_MS * CQP_TIMEOUT_THRESHOLD;

		do {
			irdma_cqp_ce_handler(rf, &rf->ccq.sc_cq);
			mdelay(1);
		} while (!sc_ah->ah_info.ah_valid && --cnt);

		if (!cnt) {
			irdma_dbg(iwdev_to_idev(iwdev),
				  "VERBS: CQP create AH timed out");
			err = -ETIMEDOUT;
			goto error;
		}
	}

copy_udata:
	if (udata) {
		uresp.ah_id = ah->sc_ah.ah_info.ah_idx;
		err = ib_copy_to_udata(udata, &uresp, sizeof(uresp));
	}
	return 0;
error:
	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);
	return err;
}
#endif

#ifdef CREATE_AH_VER_5
/**
 * irdma_create_ah - create address handle
 * @ibah: ptr to AH
 * @init_attr: address handle attributes
 * @udata: user data
 *
 * returns a pointer to an address handle
 */
int irdma_create_ah(struct ib_ah *ibah,
		    struct rdma_ah_init_attr *init_attr,
		    struct ib_udata *udata)
{
	return irdma_create_ah_v2(ibah, init_attr->ah_attr, init_attr->flags, udata);
}
#endif

#if defined(ETHER_COPY_VER_1)
void irdma_ether_copy(u8 *dmac, struct ib_ah_attr *attr)
{
	ether_addr_copy(dmac, attr->dmac);
}
#endif

#if defined(ETHER_COPY_VER_2)
void irdma_ether_copy(u8 *dmac, struct rdma_ah_attr *attr)
{
	ether_addr_copy(dmac, attr->roce.dmac);
}
#endif

#ifdef CREATE_AH_VER_1_1
/**
 * irdma_create_ah - create address handle
 * @ibpd: ptr to pd
 * @attr: address handle attributes
 * @udata: user data
 *
 * returns a pointer to an address handle
 */
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct ib_ah_attr *attr,
			      struct ib_udata *udata)
#elif defined(CREATE_AH_VER_1_2)
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      struct ib_udata *udata)
#endif
#if defined(CREATE_AH_VER_1_1) || defined(CREATE_AH_VER_1_2)
{
	struct irdma_pd *pd = to_iwpd(ibpd);
	struct irdma_device *iwdev = to_iwdev(ibpd->device);
	struct irdma_ah *ah;
#ifdef IB_GET_CACHED_GID
	union ib_gid sgid;
	struct ib_gid_attr sgid_attr;
#else
	const struct ib_gid_attr *sgid_attr;
#endif
	struct irdma_pci_f *rf = iwdev->rf;
	struct irdma_sc_ah *sc_ah;
	u32 ah_id = 0;
	struct irdma_ah_info *ah_info;
	struct irdma_create_ah_resp uresp;
	union {
		struct sockaddr saddr;
		struct sockaddr_in saddr_in;
		struct sockaddr_in6 saddr_in6;
	} sgid_addr, dgid_addr;
	int err;
	u8 dmac[ETH_ALEN];
	bool sleep;

	err = irdma_alloc_rsrc(rf, rf->allocated_ahs,
			       rf->max_ah, &ah_id, &rf->next_ah);

	if (err)
		return ERR_PTR(err);

	ah = kzalloc(sizeof(*ah), GFP_ATOMIC);
	if (!ah) {
		irdma_free_rsrc(rf, rf->allocated_ahs, ah_id);
		return ERR_PTR(-ENOMEM);
	}

	ah->pd = pd;
	sc_ah = &ah->sc_ah;
	sc_ah->ah_info.ah_idx = ah_id;
	sc_ah->ah_info.vsi = &iwdev->vsi;
	iwdev->rf->sc_dev.iw_uda_ops->init_ah(&rf->sc_dev, sc_ah);
	ah->sgid_index = attr->grh.sgid_index;
	memcpy(&ah->dgid, &attr->grh.dgid, sizeof(ah->dgid));
#ifdef IB_GET_CACHED_GID
	rcu_read_lock();
	err = ib_get_cached_gid(&iwdev->ibdev, attr->port_num,
				attr->grh.sgid_index, &sgid, &sgid_attr);
	rcu_read_unlock();
	if (err) {
		irdma_dbg(iwdev_to_idev(iwdev),
			  "VERBS: initial value: consider using coccinelle.varname: err");
		err = -EINVAL;
		goto error;
	}
	rdma_gid2ip((struct sockaddr *)&sgid_addr, &sgid);
#else
	sgid_attr = attr->grh.sgid_attr;
	rdma_gid2ip((struct sockaddr *)&sgid_addr, &sgid_attr->gid);
#endif
	rdma_gid2ip((struct sockaddr *)&dgid_addr, &attr->grh.dgid);
	ah->av.attrs = *attr;
	ah->av.net_type = kc_rdma_gid_attr_network_type(sgid_attr,
							sgid_attr.gid_type,
							&sgid);

#ifdef IB_GET_CACHED_GID
	if (kc_deref_sgid_attr(sgid_attr))
		dev_put(kc_deref_sgid_attr(sgid_attr));
#endif

	ah->av.sgid_addr.saddr = sgid_addr.saddr;
	ah->av.dgid_addr.saddr = dgid_addr.saddr;
	ah_info = &sc_ah->ah_info;
	ah_info->ah_idx = ah_id;
	ah_info->pd_idx = pd->sc_pd.pd_id;

	ether_addr_copy(ah_info->mac_addr, iwdev->netdev->dev_addr);
	if (attr->ah_flags & IB_AH_GRH) {
		ah_info->flow_label = attr->grh.flow_label;
		ah_info->hop_ttl = attr->grh.hop_limit;
		ah_info->tc_tos = attr->grh.traffic_class;
	}

	irdma_ether_copy(dmac, attr);

	if (kc_rdma_gid_attr_network_type(sgid_attr, sgid_attr.gid_type, &sgid) ==
					  RDMA_NETWORK_IPV4) {
		ah_info->ipv4_valid = true;
		ah_info->dest_ip_addr[0] =
				ntohl(dgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->src_ip_addr[0] =
				ntohl(sgid_addr.saddr_in.sin_addr.s_addr);
		ah_info->do_lpbk = irdma_ipv4_is_lpb(ah_info->src_ip_addr[0],
						     ah_info->dest_ip_addr[0]);
		if (ipv4_is_multicast(dgid_addr.saddr_in.sin_addr.s_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, true);
	} else {
		irdma_copy_ip_ntohl(ah_info->dest_ip_addr,
				    dgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		irdma_copy_ip_ntohl(ah_info->src_ip_addr,
				    sgid_addr.saddr_in6.sin6_addr.in6_u.u6_addr32);
		ah_info->do_lpbk = irdma_ipv6_is_lpb(ah_info->src_ip_addr,
						     ah_info->dest_ip_addr);
		if (rdma_is_multicast_addr(&dgid_addr.saddr_in6.sin6_addr))
			irdma_mcast_mac(ah_info->dest_ip_addr, dmac, false);
	}

	if (kc_deref_sgid_attr(sgid_attr) && is_vlan_dev(kc_deref_sgid_attr(sgid_attr)))
		ah_info->vlan_tag = vlan_dev_vlan_id(kc_deref_sgid_attr(sgid_attr));
	else
		ah_info->vlan_tag = VLAN_N_VID;

	ah_info->dst_arpindex = irdma_add_arp(iwdev->rf, ah_info->dest_ip_addr,
					      ah_info->ipv4_valid, dmac);

	if (ah_info->dst_arpindex == -1) {
		err = -EINVAL;
		goto error;
	}

	if (ah_info->vlan_tag >= VLAN_N_VID && iwdev->dcb)
		ah_info->vlan_tag = 0;

	if (ah_info->vlan_tag < VLAN_N_VID) {
		ah_info->insert_vlan_tag = true;
		ah_info->vlan_tag |=
			rt_tos2priority(ah_info->tc_tos) << VLAN_PRIO_SHIFT;
	}

	if (udata) {
		mutex_lock(&iwdev->ah_list_lock);
		if (irdma_ah_exists(iwdev, ah)) {
			mutex_unlock(&iwdev->ah_list_lock);
			irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);
			#ifdef CONFIG_DEBUG_FS
			iwdev->ah_reused++;
			#endif
			goto copy_udata;
		}
	}

	sleep = (udata ? true : false);

	err = irdma_ah_cqp_op(iwdev->rf, sc_ah, IRDMA_OP_AH_CREATE,
			      sleep, irdma_gsi_ud_qp_ah_cb, sc_ah);
	if (err) {
		if (udata) {
			if (ah->parent_ah) {
				list_del(&ah->parent_ah->list);
				kfree(ah->parent_ah);
				iwdev->ah_list_cnt--;
			}
			mutex_unlock(&iwdev->ah_list_lock);
		}
		irdma_dbg(iwdev_to_idev(iwdev),
			  "VERBS: CQP-OP Create AH fail");
		goto error;
	}

	if (udata)
		mutex_unlock(&iwdev->ah_list_lock);

	if (!sleep) {
		int cnt = CQP_COMPL_WAIT_TIME_MS * CQP_TIMEOUT_THRESHOLD;

		do {
			irdma_cqp_ce_handler(rf, &rf->ccq.sc_cq);
			mdelay(1);
		} while (!sc_ah->ah_info.ah_valid && --cnt);


		if (!cnt) {
			irdma_dbg(iwdev_to_idev(iwdev),
				  "VERBS: CQP create AH timed out");
			err = -ETIMEDOUT;
			goto error;
		}
	}

copy_udata:
	if (udata) {
		uresp.ah_id = ah->sc_ah.ah_info.ah_idx;
		err = ib_copy_to_udata(udata, &uresp, sizeof(uresp));
	}

	return &ah->ibah;
error:
	kfree(ah);
	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs, ah_id);

	return ERR_PTR(err);
}
#endif

/**
 * irdma_destroy_ah - Destroy address handle
 * @ibah: pointer to address handle
 * @ah_flags: destroy flags
 */
#if defined(DESTROY_AH_VER_4)
int irdma_destroy_ah(struct ib_ah *ibah, u32 ah_flags)
#elif defined(DESTROY_AH_VER_3)
void irdma_destroy_ah(struct ib_ah *ibah, u32 ah_flags)
#elif defined(DESTROY_AH_VER_2)
int irdma_destroy_ah(struct ib_ah *ibah, u32 ah_flags)
#elif defined(DESTROY_AH_VER_1)
int irdma_destroy_ah(struct ib_ah *ibah)
#endif
{
	struct irdma_device *iwdev = to_iwdev(ibah->device);
	struct irdma_ah *ah = to_iwah(ibah);

	mutex_lock(&iwdev->ah_list_lock);
	if (ah->parent_ah) {
		if (!refcount_dec_and_test(&ah->parent_ah->refcnt)) {
			mutex_unlock(&iwdev->ah_list_lock);
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0) && !defined(RHEL_8_2) && !defined(RHEL_8_3)
			goto done;
#elif defined(DESTROY_AH_VER_4)
			return 0;
#else
			return;
#endif
		}
		list_del(&ah->parent_ah->list);
		kfree(ah->parent_ah);
		iwdev->ah_list_cnt--;
	}
	mutex_unlock(&iwdev->ah_list_lock);

	irdma_ah_cqp_op(iwdev->rf, &ah->sc_ah, IRDMA_OP_AH_DESTROY,
			false, NULL, ah);

	irdma_free_rsrc(iwdev->rf, iwdev->rf->allocated_ahs,
			ah->sc_ah.ah_info.ah_idx);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0)
	return 0;
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0) && !defined(RHEL_8_2) && !defined(RHEL_8_3)
done:
	kfree(ah);
	return 0;
#endif
#endif
}

#ifdef SET_ROCE_CM_INFO_VER_3
int kc_irdma_set_roce_cm_info(struct irdma_qp *iwqp, struct ib_qp_attr *attr,
			      u16 *vlan_id)
{
	const struct ib_gid_attr *sgid_attr;
	int ret;
	struct irdma_av *av = &iwqp->roce_ah.av;

	sgid_attr = attr->ah_attr.grh.sgid_attr;
	if (kc_deref_sgid_attr(sgid_attr)) {
		ret = rdma_read_gid_l2_fields(sgid_attr, vlan_id,
					      iwqp->ctx_info.roce_info->mac_addr);
		if (ret)
			return ret;
	}

	rdma_gid2ip((struct sockaddr *)&av->sgid_addr, &sgid_attr->gid);
	return 0;
}
#endif

#ifdef SET_ROCE_CM_INFO_VER_2
int kc_irdma_set_roce_cm_info(struct irdma_qp *iwqp, struct ib_qp_attr *attr,
			      u16 *vlan_id)
{
	const struct ib_gid_attr *sgid_attr;
	struct irdma_av *av = &iwqp->roce_ah.av;

	sgid_attr = attr->ah_attr.grh.sgid_attr;
	if (kc_deref_sgid_attr(sgid_attr)) {
		*vlan_id = rdma_vlan_dev_vlan_id(kc_deref_sgid_attr(sgid_attr));
		ether_addr_copy(iwqp->ctx_info.roce_info->mac_addr,
				kc_deref_sgid_attr(sgid_attr)->dev_addr);
	}
	rdma_gid2ip((struct sockaddr *)&av->sgid_addr, &sgid_attr->gid);

	return 0;
}
#endif

#ifdef SET_ROCE_CM_INFO_VER_1
int kc_irdma_set_roce_cm_info(struct irdma_qp *iwqp, struct ib_qp_attr *attr,
			      u16 *vlan_id)
{
	int ret;
	union ib_gid sgid;
	struct ib_gid_attr sgid_attr;
	struct irdma_av *av = &iwqp->roce_ah.av;

	ret = ib_get_cached_gid(iwqp->ibqp.device, attr->ah_attr.port_num,
				attr->ah_attr.grh.sgid_index, &sgid,
				&sgid_attr);
	if (ret)
		return ret;

	if (kc_deref_sgid_attr(sgid_attr)) {
		*vlan_id = rdma_vlan_dev_vlan_id(kc_deref_sgid_attr(sgid_attr));
		ether_addr_copy(iwqp->ctx_info.roce_info->mac_addr,
				kc_deref_sgid_attr(sgid_attr)->dev_addr);
	}

	rdma_gid2ip((struct sockaddr *)&av->sgid_addr, &sgid);

	dev_put(kc_deref_sgid_attr(sgid_attr));

	return 0;
}

#endif
#ifdef IRDMA_DESTROY_CQ_VER_4
/**
 * irdma_destroy_cq - destroy cq
 * @ib_cq: cq pointer
 * @udata: user data
 */
int irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(ib_cq->device);
	struct irdma_cq *iwcq = to_iwcq(ib_cq);
	struct irdma_sc_cq *cq = &iwcq->sc_cq;
	struct irdma_sc_dev *dev = cq->dev;
	struct irdma_sc_ceq *ceq = dev->ceq[cq->ceq_id];
	struct irdma_ceq *iwceq = container_of(ceq, struct irdma_ceq, sc_ceq);
	unsigned long flags;

	if (!list_empty(&iwcq->resize_list)) {
		spin_lock_irqsave(&iwcq->lock, flags);
		irdma_process_resize_list(iwcq, iwdev, NULL);
		spin_unlock_irqrestore(&iwcq->lock, flags);
	}
	irdma_cq_wq_destroy(iwdev->rf, cq);
	irdma_cq_free_rsrc(iwdev->rf, iwcq);

	spin_lock_irqsave(&iwceq->ce_lock, flags);
	dev->ceq_ops->cleanup_ceqes(cq, ceq);
	spin_unlock_irqrestore(&iwceq->ce_lock, flags);

	return 0;
}

#endif /* IRDMA_DESTROY_CQ_VER_4 */
#ifdef IRDMA_DESTROY_CQ_VER_3
/**
 * irdma_destroy_cq - destroy cq
 * @ib_cq: cq pointer
 * @udata: user data
 */
void irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(ib_cq->device);
	struct irdma_cq *iwcq = to_iwcq(ib_cq);
	struct irdma_sc_cq *cq = &iwcq->sc_cq;
	struct irdma_sc_dev *dev = cq->dev;
	struct irdma_sc_ceq *ceq = dev->ceq[cq->ceq_id];
	struct irdma_ceq *iwceq = container_of(ceq, struct irdma_ceq, sc_ceq);
	unsigned long flags;

	if (!list_empty(&iwcq->resize_list)) {
		spin_lock_irqsave(&iwcq->lock, flags);
		irdma_process_resize_list(iwcq, iwdev, NULL);
		spin_unlock_irqrestore(&iwcq->lock, flags);
	}
	irdma_cq_wq_destroy(iwdev->rf, cq);
	irdma_cq_free_rsrc(iwdev->rf, iwcq);

	spin_lock_irqsave(&iwceq->ce_lock, flags);
	dev->ceq_ops->cleanup_ceqes(cq, ceq);
	spin_unlock_irqrestore(&iwceq->ce_lock, flags);
}

#endif /* IRDMA_DESTROY_CQ_VER_3 */
#ifdef IRDMA_DESTROY_CQ_VER_2
/**
 * irdma_destroy_cq - destroy cq
 * @ib_cq: cq pointer
 * @udata: user data
 */
int irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(ib_cq->device);
	struct irdma_cq *iwcq = to_iwcq(ib_cq);
	struct irdma_sc_cq *cq = &iwcq->sc_cq;
	struct irdma_sc_dev *dev = cq->dev;
	struct irdma_sc_ceq *ceq = dev->ceq[cq->ceq_id];
	struct irdma_ceq *iwceq = container_of(ceq, struct irdma_ceq, sc_ceq);
	unsigned long flags;

	if (!list_empty(&iwcq->resize_list)) {
		spin_lock_irqsave(&iwcq->lock, flags);
		irdma_process_resize_list(iwcq, iwdev, NULL);
		spin_unlock_irqrestore(&iwcq->lock, flags);
	}
	irdma_cq_wq_destroy(iwdev->rf, cq);
	irdma_cq_free_rsrc(iwdev->rf, iwcq);

	spin_lock_irqsave(&iwceq->ce_lock, flags);
	dev->ceq_ops->cleanup_ceqes(cq, ceq);
	spin_unlock_irqrestore(&iwceq->ce_lock, flags);

	kfree(iwcq);

	return 0;
}

#endif /* IRDMA_DESTROY_CQ_VER_2 */
#ifdef IRDMA_DESTROY_CQ_VER_1
/**
 * irdma_destroy_cq - destroy cq
 * @ib_cq: cq pointer
 */
int irdma_destroy_cq(struct ib_cq *ib_cq)
{
	struct irdma_device *iwdev = to_iwdev(ib_cq->device);
	struct irdma_cq *iwcq = to_iwcq(ib_cq);
	struct irdma_sc_cq *cq = &iwcq->sc_cq;
	struct irdma_sc_dev *dev = cq->dev;
	struct irdma_sc_ceq *ceq = dev->ceq[cq->ceq_id];
	struct irdma_ceq *iwceq = container_of(ceq, struct irdma_ceq, sc_ceq);
	unsigned long flags;

	if (!list_empty(&iwcq->resize_list)) {
		spin_lock_irqsave(&iwcq->lock, flags);
		irdma_process_resize_list(iwcq, iwdev, NULL);
		spin_unlock_irqrestore(&iwcq->lock, flags);
	}
	irdma_cq_wq_destroy(iwdev->rf, cq);
	irdma_cq_free_rsrc(iwdev->rf, iwcq);

	spin_lock_irqsave(&iwceq->ce_lock, flags);
	dev->ceq_ops->cleanup_ceqes(cq, ceq);
	spin_unlock_irqrestore(&iwceq->ce_lock, flags);

	kfree(iwcq);

	return 0;
}

#endif /* IRDMA_DESTROY_CQ_VER_1 */

#ifdef IRDMA_ALLOC_MW_VER_2
/**
 * irdma_alloc_mw - Allocate memory window
 * @ibmw: Memory Window
 * @udata: user data pointer
 */
int irdma_alloc_mw(struct ib_mw *ibmw, struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(ibmw->device);
	struct irdma_mr *iwmr = to_iwmw(ibmw);
	int err_code;
	u32 stag;

	stag = irdma_create_stag(iwdev);
	if (!stag)
		return -ENOMEM;

	iwmr->stag = stag;
	ibmw->rkey = stag;
	iwmr->type = IW_MEMREG_TYPE_MW;

	err_code = irdma_hw_alloc_mw(iwdev, iwmr);
	if (err_code) {
		irdma_free_stag(iwdev, stag);
		return err_code;
	}

	return 0;
}

#endif /* IRDMA_ALLOC_MW_VER_2 */
#ifdef IRDMA_ALLOC_MW_VER_1
/**
 * irdma_alloc_mw - Allocate memory window
 * @pd: Protection domain
 * @type: Window type
 * @udata: user data pointer
 */
struct ib_mw *irdma_alloc_mw(struct ib_pd *pd, enum ib_mw_type type,
			     struct ib_udata *udata)
{
	struct irdma_device *iwdev = to_iwdev(pd->device);
	struct irdma_mr *iwmr;
	int err_code;
	u32 stag;

	iwmr = kzalloc(sizeof(*iwmr), GFP_KERNEL);
	if (!iwmr)
		return ERR_PTR(-ENOMEM);

	stag = irdma_create_stag(iwdev);
	if (!stag) {
		kfree(iwmr);
		return ERR_PTR(-ENOMEM);
	}

	iwmr->stag = stag;
	iwmr->ibmw.rkey = stag;
	iwmr->ibmw.pd = pd;
	iwmr->ibmw.type = type;
	iwmr->ibmw.device = pd->device;
	iwmr->type = IW_MEMREG_TYPE_MW;

	err_code = irdma_hw_alloc_mw(iwdev, iwmr);
	if (err_code) {
		irdma_free_stag(iwdev, stag);
		kfree(iwmr);
		return ERR_PTR(err_code);
	}

	return &iwmr->ibmw;
}

#endif /* IRDMA_ALLOC_MW_VER_1 */

#ifdef DEVLINK_SUPPORTED
#ifdef DEVLINK_RELOAD_DOWN_VER_3
int irdma_devlink_reload_down(struct devlink *devlink, bool netns_change,
			      enum devlink_reload_action action,
			      enum devlink_reload_limit limit,
			      struct netlink_ext_ack *extack)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);

#ifdef DEVLINK_RELOAD_DOWN_NETNS_CHANGE
	if (netns_change) {
		NL_SET_ERR_MSG_MOD(extack, "Namespace change is not supported");
		return -EOPNOTSUPP;
	}

#endif
	priv->drvdata->deinit_dev(pdev);

	return 0;
}

#endif

#ifdef DEVLINK_RELOAD_DOWN_VER_2
int irdma_devlink_reload_down(struct devlink *devlink, bool netns_change,
			      struct netlink_ext_ack *extack)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);

#ifdef DEVLINK_RELOAD_DOWN_NETNS_CHANGE
	if (netns_change) {
		NL_SET_ERR_MSG_MOD(extack, "Namespace change is not supported");
		return -EOPNOTSUPP;
	}

#endif
	priv->drvdata->deinit_dev(pdev);

	return 0;
}

#endif

#ifdef DEVLINK_RELOAD_DOWN_VER_1
int irdma_devlink_reload_down(struct devlink *devlink,
			      struct netlink_ext_ack *extack)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);

#ifdef DEVLINK_RELOAD_DOWN_NETNS_CHANGE
	if (netns_change) {
		NL_SET_ERR_MSG_MOD(extack, "Namespace change is not supported");
		return -EOPNOTSUPP;
	}

#endif
	priv->drvdata->deinit_dev(pdev);

	return 0;
}

#endif

#ifdef DEVLINK_RELOAD_UP_VER_3 /*(LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0))*/
int irdma_devlink_reload_up(struct devlink *devlink, enum devlink_reload_action action,
			    enum devlink_reload_limit limit, u32 *actions_performed,
			    struct netlink_ext_ack *extack)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);
	union devlink_param_value saved_value;
	int ret;

	*actions_performed = BIT(DEVLINK_RELOAD_ACTION_DRIVER_REINIT);

#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
					   &saved_value);
#else
	devlink_param_driverinit_value_get(devlink,
					   DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE,
					   &saved_value);
#endif
	priv->roce_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
					   &saved_value);
	priv->limits_sel = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
					   &saved_value);
	priv->dcqcn_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
					   &saved_value);
	priv->dcqcn_params.cc_cfg_valid = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
					   &saved_value);
	priv->dcqcn_params.min_dec_factor = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_RATE, &saved_value);
	priv->dcqcn_params.min_rate = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_F, &saved_value);
	priv->dcqcn_params.dcqcn_f = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_T, &saved_value);
	priv->dcqcn_params.dcqcn_t = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_B, &saved_value);
	priv->dcqcn_params.dcqcn_b = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.rai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
					   &saved_value);
	priv->fragcnt_limit = saved_value.vu8;

#ifndef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	priv->drvdata->deinit_dev(pdev);
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
	priv->reload = true;
	ret = priv->drvdata->init_dev(pdev);
	priv->reload = false;
	return ret;
}
#endif

#ifdef DEVLINK_RELOAD_UP_VER_2 /*defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)*/
int irdma_devlink_reload_up(struct devlink *devlink)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);
	union devlink_param_value saved_value;
	int ret;

#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
					   &saved_value);
#else
	devlink_param_driverinit_value_get(devlink,
					   DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE,
					   &saved_value);
#endif
	priv->roce_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
					   &saved_value);
	priv->limits_sel = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
					   &saved_value);
	priv->dcqcn_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
					   &saved_value);
	priv->dcqcn_params.cc_cfg_valid = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
					   &saved_value);
	priv->dcqcn_params.min_dec_factor = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_RATE, &saved_value);
	priv->dcqcn_params.min_rate = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_F, &saved_value);
	priv->dcqcn_params.dcqcn_f = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_T, &saved_value);
	priv->dcqcn_params.dcqcn_t = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_B, &saved_value);
	priv->dcqcn_params.dcqcn_b = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.rai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
					   &saved_value);
	priv->fragcnt_limit = saved_value.vu8;

#ifndef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	priv->drvdata->deinit_dev(pdev);
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
	priv->reload = true;
	ret = priv->drvdata->init_dev(pdev);
	priv->reload = false;
	return ret;
}

#endif

#ifdef DEVLINK_RELOAD_UP_VER_1
int irdma_devlink_reload_up(struct devlink *devlink,
			    struct netlink_ext_ack *extack)
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	struct platform_device *pdev = to_platform_device(devlink->dev);
	union devlink_param_value saved_value;
	int ret;

#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
					   &saved_value);
#else
	devlink_param_driverinit_value_get(devlink,
					   DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE,
					   &saved_value);
#endif
	priv->roce_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
					   &saved_value);
	priv->limits_sel = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
					   &saved_value);
	priv->dcqcn_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
					   &saved_value);
	priv->dcqcn_params.cc_cfg_valid = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
					   &saved_value);
	priv->dcqcn_params.min_dec_factor = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_MIN_RATE, &saved_value);
	priv->dcqcn_params.min_rate = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_F, &saved_value);
	priv->dcqcn_params.dcqcn_f = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_T, &saved_value);
	priv->dcqcn_params.dcqcn_t = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_DCQCN_B, &saved_value);
	priv->dcqcn_params.dcqcn_b = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.rai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
					   &saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
					   IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
					   &saved_value);
	priv->fragcnt_limit = saved_value.vu8;

#ifndef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	priv->drvdata->deinit_dev(pdev);
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
	priv->reload = true;
	ret = priv->drvdata->init_dev(pdev);
	priv->reload = false;
	return ret;
}
#endif

#endif /* DEVLINK_SUPPORTED */
/**
 * kc_set_loc_seq_num_mss - Set local seq number and mss
 * @cm_node: cm node info
 */
void kc_set_loc_seq_num_mss(struct irdma_cm_node *cm_node)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 19, 0)
	if (cm_node->ipv4) {
		cm_node->tcp_cntxt.loc_seq_num = secure_tcp_seq(htonl(cm_node->loc_addr[0]),
								htonl(cm_node->rem_addr[0]),
								htons(cm_node->loc_port),
								htons(cm_node->rem_port));
		if (cm_node->iwdev->vsi.mtu > 1500 &&
		    2 * cm_node->iwdev->vsi.mtu > cm_node->iwdev->rcv_wnd)
			cm_node->tcp_cntxt.mss = 1500 - IRDMA_MTU_TO_MSS_IPV4;
		else
			cm_node->tcp_cntxt.mss = cm_node->iwdev->vsi.mtu - IRDMA_MTU_TO_MSS_IPV4;
	} else if (IS_ENABLED(CONFIG_IPV6)) {
		__be32 loc[4] = {
			htonl(cm_node->loc_addr[0]), htonl(cm_node->loc_addr[1]),
			htonl(cm_node->loc_addr[2]), htonl(cm_node->loc_addr[3])
		};
		__be32 rem[4] = {
			htonl(cm_node->rem_addr[0]), htonl(cm_node->rem_addr[1]),
			htonl(cm_node->rem_addr[2]), htonl(cm_node->rem_addr[3])
		};
		cm_node->tcp_cntxt.loc_seq_num = secure_tcpv6_seq(loc, rem,
								  htons(cm_node->loc_port),
								  htons(cm_node->rem_port));
		if (cm_node->iwdev->vsi.mtu > 1500 &&
		    2 * cm_node->iwdev->vsi.mtu > cm_node->iwdev->rcv_wnd)
			cm_node->tcp_cntxt.mss = 1500 - IRDMA_MTU_TO_MSS_IPV6;
		else
			cm_node->tcp_cntxt.mss = cm_node->iwdev->vsi.mtu - IRDMA_MTU_TO_MSS_IPV6;
	}
#else
	cm_node->tcp_cntxt.loc_seq_num = htonl(current_kernel_time().tv_nsec);
	if (cm_node->iwdev->vsi.mtu > 1500 &&
	    2 * cm_node->iwdev->vsi.mtu > cm_node->iwdev->rcv_wnd)
		cm_node->tcp_cntxt.mss = (cm_node->ipv4) ?
					 (1500 - IRDMA_MTU_TO_MSS_IPV4) :
					 (1500 - IRDMA_MTU_TO_MSS_IPV6);
	else
		cm_node->tcp_cntxt.mss = (cm_node->ipv4) ?
					 (cm_node->iwdev->vsi.mtu - IRDMA_MTU_TO_MSS_IPV4) :
					 (cm_node->iwdev->vsi.mtu - IRDMA_MTU_TO_MSS_IPV6);
#endif
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 20, 0) && !defined(RHEL_8_2) && !defined(RHEL_8_3)
struct irdma_vma_data {
	struct list_head list;
	struct vm_area_struct *vma;
	struct mutex *vma_list_mutex; /* protect the vma_list */
};

/**
 * irdma_vma_open -
 * @vma: User VMA
 */
static void irdma_vma_open(struct vm_area_struct *vma)
{
	vma->vm_ops = NULL;
}

/**
 * irdma_vma_close - Remove vma data from vma list
 * @vma: User VMA
 */
static void irdma_vma_close(struct vm_area_struct *vma)
{
	struct irdma_vma_data *vma_data;

	vma_data = vma->vm_private_data;
	vma->vm_private_data = NULL;
	vma_data->vma = NULL;
	mutex_lock(vma_data->vma_list_mutex);
	list_del(&vma_data->list);
	mutex_unlock(vma_data->vma_list_mutex);
	kfree(vma_data);
}

static const struct vm_operations_struct irdma_vm_ops = {
	.open = irdma_vma_open,
	.close = irdma_vma_close
};

/**
 * irdma_set_vma_data - Save vma data in context list
 * @vma: User VMA
 * @context: ib user context
 */
static int irdma_set_vma_data(struct vm_area_struct *vma,
			      struct irdma_ucontext *context)
{
	struct list_head *vma_head = &context->vma_list;
	struct irdma_vma_data *vma_entry;

	vma_entry = kzalloc(sizeof(*vma_entry), GFP_KERNEL);
	if (!vma_entry)
		return -ENOMEM;

	vma->vm_private_data = vma_entry;
	vma->vm_ops = &irdma_vm_ops;

	vma_entry->vma = vma;
	vma_entry->vma_list_mutex = &context->vma_list_mutex;

	mutex_lock(&context->vma_list_mutex);
	list_add(&vma_entry->list, vma_head);
	mutex_unlock(&context->vma_list_mutex);

	return 0;
}

/**
 * irdma_disassociate_ucontext - Disassociate user context
 * @context: ib user context
 */
void irdma_disassociate_ucontext(struct ib_ucontext *context)
{
	struct irdma_ucontext *ucontext = to_ucontext(context);

	struct irdma_vma_data *vma_data, *n;
	struct vm_area_struct *vma;

	mutex_lock(&ucontext->vma_list_mutex);
	list_for_each_entry_safe(vma_data, n, &ucontext->vma_list, list) {
		vma = vma_data->vma;
		zap_vma_ptes(vma, vma->vm_start, PAGE_SIZE);

		vma->vm_flags &= ~(VM_SHARED | VM_MAYSHARE);
		vma->vm_ops = NULL;
		list_del(&vma_data->list);
		kfree(vma_data);
	}
	mutex_unlock(&ucontext->vma_list_mutex);
}

int rdma_user_mmap_io(struct ib_ucontext *context, struct vm_area_struct *vma,
		      unsigned long pfn, unsigned long size, pgprot_t prot)
{
	if (io_remap_pfn_range(vma,
			       vma->vm_start,
			       pfn,
			       size,
			       prot))
			return -EAGAIN;

	return irdma_set_vma_data(vma, to_ucontext(context));
}
#else
/**
 * irdma_disassociate_ucontext - Disassociate user context
 * @context: ib user context
 */
void irdma_disassociate_ucontext(struct ib_ucontext *context)
{
}
#endif /* < 4.20.0 */

struct irdma_device *kc_irdma_get_device(struct net_device *netdev)
{
#ifndef NETDEV_TO_IBDEV_SUPPORT
struct irdma_device *iwdev;
	struct irdma_handler *hdl;
	struct list_head *pos;
	struct list_head *tmp;
	unsigned long flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_for_each_entry(hdl, &irdma_handlers, list) {
		list_for_each_safe(pos, tmp, &hdl->rf.vsi_dev_list) {
			iwdev = container_of(pos, struct irdma_device, list);
			if (netdev == iwdev->netdev) {
				spin_unlock_irqrestore(&irdma_handler_lock,
						       flags);
				return iwdev;
			}
		}
	}
	spin_unlock_irqrestore(&irdma_handler_lock, flags);

	return NULL;
#else
	struct ib_device *ibdev = ib_device_get_by_netdev(netdev,
							  RDMA_DRIVER_I40IW);

	if (!ibdev)
		return NULL;

	return to_iwdev(ibdev);
#endif /* NETDEV_TO_IBDEV_SUPPORT */
}

void kc_irdma_put_device(struct irdma_device *iwdev)
{
#ifndef NETDEV_TO_IBDEV_SUPPORT
	return;
#else
	struct ib_device *ibdev = &iwdev->ibdev;

	ib_device_put(ibdev);
#endif
}

/**
 * irdma_modify_port - modify port attributes
 * @ibdev: device pointer from stack
 * @port: port number for query
 * @mask: Property mask
 * @props: returning device attributes
 */
int irdma_modify_port(struct ib_device *ibdev, u8 port, int mask,
		      struct ib_port_modify *props)
{
	if (port > 1)
		return -EINVAL;

	return 0;
}
