// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2015 - 2021 Intel Corporation */
#include "main.h"
#include "i40iw_hw.h"
#include "i40e_client.h"

/**
 * i40iw_request_reset - Request a reset
 * @rf: RDMA PCI function
 *
 */
static void i40iw_request_reset(struct irdma_pci_f *rf)
{
	struct i40e_info *peer_info = rf->priv_peer_info.peer_info;

	peer_info->ops->request_reset(peer_info, rf->priv_peer_info.client, 1);
}

/**
 * i40iw_open - client interface operation open for iwarp/uda device
 * @peer_info: parent lan device information structure with data/ops
 * @client: iwarp client information, provided during registration
 *
 * Called by the lan driver during the processing of client register
 * Create device resources, set up queues, pble and hmc objects and
 * register the device with the ib verbs interface
 * Return 0 if successful, otherwise return error
 */
static int i40iw_open(struct i40e_info *peer_info, struct i40e_client *client)
{
	struct irdma_device *iwdev = NULL;
	struct irdma_handler *hdl = NULL;
	struct irdma_priv_peer_info *priv_peer_info;
	struct irdma_sc_dev *dev;
	struct irdma_pci_f *rf;
	struct irdma_l2params l2params = {};
	int err = -EIO;
	int i;
	u16 qset;
	u16 last_qset = IRDMA_NO_QSET;

	hdl = irdma_find_handler(peer_info->pcidev);
	if (hdl)
		return 0;

	hdl = kzalloc(sizeof(*hdl), GFP_KERNEL);
	if (!hdl)
		return -ENOMEM;

	rf = &hdl->rf;
	rf->hdl = hdl;
	dev = &rf->sc_dev;
	dev->back_dev = rf;
	rf->rdma_ver = IRDMA_GEN_1;
	rf->pdev = peer_info->platform_dev;
	irdma_set_config_params(rf);
	rf->gen_ops.init_hw = i40iw_init_hw;
	rf->gen_ops.request_reset = i40iw_request_reset;
	rf->hw.hw_addr = peer_info->hw_addr;
	rf->pcidev = peer_info->pcidev;
	rf->netdev = peer_info->netdev;
	dev->pci_rev = rf->pcidev->revision;

	priv_peer_info = &rf->priv_peer_info;
	priv_peer_info->client = client;
	priv_peer_info->peer_info = peer_info;
	priv_peer_info->fn_num = peer_info->fid;
	priv_peer_info->ftype = peer_info->ftype;
	priv_peer_info->pf_vsi_num = 0;
	priv_peer_info->msix_count = peer_info->msix_count;
	priv_peer_info->msix_entries = peer_info->msix_entries;

	if (irdma_ctrl_init_hw(rf)) {
		err = -EIO;
		goto err_ctrl_init;
	}

	iwdev = ib_alloc_device(irdma_device, ibdev);
	if (!iwdev) {
		err = -ENOMEM;
		goto err_ib_alloc;
	}

	iwdev->rf = rf;
	iwdev->hdl = hdl;
	iwdev->init_state = INITIAL_STATE;
	iwdev->rcv_wnd = IRDMA_CM_DEFAULT_RCV_WND_SCALED;
	iwdev->rcv_wscale = IRDMA_CM_DEFAULT_RCV_WND_SCALE;
	iwdev->netdev = peer_info->netdev;
	iwdev->vsi_num = 0;

	l2params.mtu =
		(peer_info->params.mtu) ? peer_info->params.mtu : IRDMA_DEFAULT_MTU;
	for (i = 0; i < I40E_CLIENT_MAX_USER_PRIORITY; i++) {
		qset = peer_info->params.qos.prio_qos[i].qs_handle;
		l2params.up2tc[i] = peer_info->params.qos.prio_qos[i].tc;
		l2params.qs_handle_list[i] = qset;
		if (last_qset == IRDMA_NO_QSET)
			last_qset = qset;
		else if ((qset != last_qset) && (qset != IRDMA_NO_QSET))
			iwdev->dcb = true;
	}

	if (irdma_rt_init_hw(rf, iwdev, &l2params)) {
		err = -EIO;
		goto err_rt_init;
	}

	err = irdma_ib_register_device(iwdev);
	if (err)
		goto err_ibreg;

	irdma_add_handler(hdl);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_init(hdl);
#endif
	irdma_dbg(dev, "INIT: Gen1 VSI open success peer_info=%p\n",
		  peer_info);

	return 0;

err_ibreg:
	irdma_rt_deinit_hw(iwdev);
err_rt_init:
	ib_dealloc_device(&iwdev->ibdev);
err_ib_alloc:
	irdma_ctrl_deinit_hw(rf);
err_ctrl_init:
	kfree(hdl);

	return err;
}

/**
 * i40iw_l2param_change - handle mss change
 * @peer_info: parent lan device information structure with data/ops
 * @client: client for parameter change
 * @params: new parameters from L2
 */
static void i40iw_l2param_change(struct i40e_info *peer_info,
				 struct i40e_client *client,
				 struct i40e_params *params)
{
	struct irdma_l2params l2params = {};
	struct irdma_device *iwdev;

	iwdev = irdma_get_device(peer_info->netdev);
	if (!iwdev)
		return;

	if (iwdev->vsi.mtu != params->mtu) {
		l2params.mtu_changed = true;
		l2params.mtu = params->mtu;
	}
	irdma_change_l2params(&iwdev->vsi, &l2params);
	irdma_put_device(iwdev);
}

/**
 * i40iw_close - client interface operation close for iwarp/uda device
 * @peer_info: parent lan device information structure with data/ops
 * @client: client to close
 * @reset: flag to indicate close on reset
 *
 * Called by the lan driver during the processing of client unregister
 * Destroy and clean up the driver resources
 */
static void i40iw_close(struct i40e_info *peer_info, struct i40e_client *client,
			bool reset)
{
	struct irdma_handler *hdl;
	struct irdma_pci_f *rf;
	struct irdma_device *iwdev;

	hdl = irdma_find_handler(peer_info->pcidev);
	if (!hdl)
		return;

	rf = &hdl->rf;
	iwdev = list_first_entry_or_null(&rf->vsi_dev_list, struct irdma_device,
					 list);
	if (!iwdev)
		return;

	if (reset)
		iwdev->reset = true;

	irdma_ib_unregister_device(iwdev);
#ifndef IB_DEALLOC_DRIVER_SUPPORT
	irdma_rt_deinit_hw(iwdev);
	ib_dealloc_device(&iwdev->ibdev);
#endif
	irdma_deinit_rf(rf);
	pr_debug("INIT: Gen1 VSI close complete peer_info=%p\n", peer_info);
}

/* client interface functions */
static const struct i40e_client_ops i40e_ops = {
	.open = i40iw_open,
	.close = i40iw_close,
	.l2_param_change = i40iw_l2param_change
};

static struct i40e_client i40iw_client = {
	.ops = &i40e_ops,
	.version.major = I40E_CLIENT_VERSION_MAJOR,
	.version.minor = I40E_CLIENT_VERSION_MINOR,
	.version.build = I40E_CLIENT_VERSION_BUILD,
	.type = I40E_CLIENT_IWARP,
};

int i40iw_init_dev(struct platform_device *pdev)
{
	struct i40e_peer_dev_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct i40e_info *peer_info = pdata->ldev;

	if (peer_info->version.major != I40E_CLIENT_VERSION_MAJOR ||
	    peer_info->version.minor != I40E_CLIENT_VERSION_MINOR) {
		pr_err("version mismatch:\n");
		pr_err("expected major ver %d, caller specified major ver %d\n",
		       I40E_CLIENT_VERSION_MAJOR, peer_info->version.major);
		pr_err("expected minor ver %d, caller specified minor ver %d\n",
		       I40E_CLIENT_VERSION_MINOR, peer_info->version.minor);
		return -EINVAL;
	}

	strncpy(i40iw_client.name, "irdma", I40E_CLIENT_STR_LENGTH);
	peer_info->client = &i40iw_client;
	peer_info->platform_dev = pdev;

	return peer_info->ops->client_device_register(peer_info);
}

int i40iw_deinit_dev(struct platform_device *pdev)
{
	struct i40e_peer_dev_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct i40e_info *peer_info = pdata->ldev;

	peer_info->ops->client_device_unregister(peer_info);

	return 0;
}

